(ns cljs.com.edocu.communication.aws-iot
  (:require [cljs.nodejs :as nodejs]
            [com.stuartsierra.component :as component]
            [cljs.core.async :refer [<! promise-chan put! close! chan]]
            [cljs.com.edocu.log :refer [debug error prepare-logger]]
            [cljs.com.edocu.utils :refer [stringify parse-json]]
            [com.edocu.communication.protocols :as prots]
            [clojure.string :as clj-str]
            [cljs.com.edocu.communication.kafka.edocu-config :as edocu-config])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defonce aws-iot (nodejs/require "aws-iot-device-sdk"))

(defn- ->mqtt-topic
  [topic]
  (-> topic
      (clj-str/replace "." "/")
      (clj-str/replace "*" "+")))

(defn- ->edocu-topic
  [topic]
  (clj-str/replace topic "/" "."))

(defprotocol IDeviceFactory
  (<device [this] "Return AWS Device"))

(defrecord Communicator []
  component/Lifecycle
  (start [this]
    (prepare-logger)
    (debug "Communicator component started")
    (let [e (nodejs/require "dotenv")]
      (.config e))
    this)

  (stop [this]
    (debug "Communicator component stopped")
    this)

  IDeviceFactory
  (<device [_]
    (let [>result (chan)
          d (.device
              aws-iot
              (clj->js {:accessKeyId (.-AWS_ACCESS_KEY_ID (.-env nodejs/process))
                        :secretKey   (.-AWS_SECRET_ACCESS_KEY (.-env nodejs/process))
                        :protocol    "wss"
                        :clientId    (str (random-uuid))
                        :host        (.-IOT_HOST (.-env nodejs/process))}))]
      (.on d "connect"
           (fn []
             (debug "Connected")
             (put! >result d)))
      (.on d "error"
           (fn [err]
             (error "AWS IoT" err)))
      >result))

  prots/IMessageManagement
  (send-message! [this topic message]
    (let [>result (chan)]
      (go
        (let [d (<! (<device this))
              topic (->mqtt-topic topic)
              body (stringify message)]
          (debug "send-message. topic:" topic "body:" body)
          (.publish
            d
            topic
            body
            (clj->js {:qos 1})
            (fn [err]
              (debug "message send.")
              (if err
                (error "cannot send me"))
              (.end d false (fn [] (close! >result)))))))
      >result))

  (deliver-message! [this _ original-topic message]
    (prots/send-message!
      this
      original-topic
      message))

  prots/ITopicManagement
  (register-topics! [this topics]
    (try
      (if (seq topics)
        (let [topics_str (clojure.string/join "," topics)]
          (prots/send-message!
            this
            (edocu-config/register-topic)
            {:topic topics_str})))
      (catch :default err
        (error "register-topics!"
               "topics:" topics
               "error:" err))))

  (subscribe-to-topic [this topic]
    (let [>callback (chan)]
      (go
        (let [d (<! (<device this))]
          (.subscribe d (->mqtt-topic topic))
          (.on d "message"
               (fn [topic payload]
                 (try
                   (let [payload (parse-json (.toString payload))]
                     (if (some? payload)
                       (put! >callback {:key   (->edocu-topic topic)
                                        :value payload})))
                   (catch :default e
                     (error "Invalid message payload. error:" e)))))))
      >callback))

  prots/IErrorsManagement
  (send-malformed-message-report!
    [this message]
    (prots/send-message!
      this
      "system.errors.message.malformed"
      message))

  (send-service-error-report!
    [this message]
    (prots/send-message!
      this
      "system.errors.service"
      message)))

(defn make->communicator
  []
  (map->Communicator {}))

(comment
  (let [system (component/system-map
                 :communicator (map->Communicator {}))
        {:keys [communicator]} (component/start-system system)
        subscription (prots/subscribe-to-topic communicator "elements.*")]
    (go
      (loop [msg (<! subscription)]
        (println "MSG:" (pr-str msg))
        (if (some? msg)
          (recur (<! subscription)))))
    (prots/send-message! communicator "elements.ahoj" {:body "body"})
    (prots/send-message! communicator "elements.cau" {:body "telo"})))


