(ns e85th.ui.rf.awesomplete
  (:require [re-frame.core :as rf]
            [reagent.core :as reagent]
            [reagent.ratom :as ratom]
            [taoensso.timbre :as log]
            [devcards.core :as d :refer-macros [defcard defcard-rg]]
            [e85th.ui.dom :as dom]
            [e85th.ui.util :as u]))

(defn new-instance
  "Creates a new instance based on the dom-sel and opts map."
  [dom-sel opts]
  (js/Awesomplete. (.querySelector js/document dom-sel) (clj->js opts)))


(defn set-list
  "Sets the list property for the awesomplete instance. xs are convereted to js objs."
  [inst xs]
  (set! (.-list inst) (clj->js xs)))

(defn evaluate
  [inst]
  (.evaluate inst))

(defn close
  [inst]
  (.close inst))

(defn open
  [inst]
  (.open inst))

(defn destroy
  [inst]
  (when inst
    (.destroy inst)))

(defn- comboplete-on-click
  [combo]
  (let [len (some-> combo .-ul .-childNodes .-length)]
    (if (and len (zero? len))
      (do
        (set! (.-minChars combo) 0)
        (evaluate combo))
      (if (some-> combo .-ul (.hasAttribute "hidden"))
        (open combo)
        (close combo)))))


(defn comboplete
  [selected-sub options-sub selection-event opts]
  (let [dom-id (str (gensym "comboplete-"))
        button-dom-id (str dom-id "-button")
        comboplete-atom (atom nil)
        first-run-data (atom {:items []
                              :selection ""})]
    (reagent/create-class
     {:display-name "comboplete"
      :reagent-render (fn [selected-sub options-sub selection-event opts] ;; <-- use these because the subscriptions actually change otherwise data won't update
                        (let [items @(rf/subscribe (u/as-vector options-sub))
                              selection @(rf/subscribe (u/as-vector selected-sub))
                              comboplete-instance @comboplete-atom]
                          ;(log/infof "suggested items before when %s, dom-id %s" items dom-id)
                          (if comboplete-instance
                            (do
                              (dom/set-element-value dom-id (or selection ""))
                              ;; must call after setting list property to regenerate the list
                              ;; close it otherwise the box is open with the selected item highlighted
                              (doto comboplete-instance
                                (set-list items)
                                (evaluate)
                                (close)))

                            (reset! first-run-data {:items items
                                                    :selection selection})))
                        ;; always return the same markup for react
                        [:span [:button {:id button-dom-id} "V"] [:input {:id dom-id}]])
      :component-did-mount (fn []
                             (reset! comboplete-atom (new-instance (str "#" dom-id) opts))
                             (let [{:keys [items selection] :or {:items [] :selection ""}} @first-run-data]
                               (set-list @comboplete-atom items)
                               (dom/set-element-value dom-id selection))
                             ;(log/infof "comboplete opts were %s, first-run data %s" opts @first-run-data)
                             (dom/add-event-listener button-dom-id "click" #(comboplete-on-click @comboplete-atom)))
      :component-will-unmount #(destroy @comboplete-atom)})))
