(ns e85th.gaia.core
  "Contains code mostly taken from https://gist.github.com/damionjunk/3757473."
  (:require [clojure.java.io :as io]
            [com.stuartsierra.component :as component]
            [taoensso.timbre :as log]
            [schema.core :as s])
  (:import [org.geotools.data DataStore DataStoreFinder FileDataStore FileDataStoreFinder Query]
           [org.geotools.data.shapefile ShapefileDataStoreFactory]
           [org.geotools.feature FeatureIterator]
           [org.geotools.factory CommonFactoryFinder Hints]
           [org.geotools.geometry.jts JTSFactoryFinder]
           [org.geotools.geometry DirectPosition2D]
           [org.geotools.filter.text.cql2 CQL]
           [org.opengis.feature.simple SimpleFeature]
           [com.vividsolutions.jts.geom Point Coordinate MultiPolygon]
           [java.io File]))

(defn iteration-seq
  [^FeatureIterator iter]
  (iterator-seq
   (reify java.util.Iterator
     (hasNext [this] (.hasNext iter))
     (next [this] (.next iter))
     (remove [this] (UnsupportedOperationException. "Remove not suported.")))))

(defn locate*
  [^FileDataStore data-store lat lng]
  (let [geo-filter (CQL/toFilter (format "CONTAINS(the_geom, POINT(%s %s))" lng lat))
        q (doto (Query.)
            (.setFilter geo-filter)
            (.setHints (Hints. Hints/FEATURE_2D Boolean/TRUE)))
        feature-iter (volatile! (-> data-store .getFeatureSource (.getFeatures q) .features))]
    (try
      (into [] (iteration-seq @feature-iter))
      (finally
        (some-> ^FeatureIterator @feature-iter .close)))))

(defn shape-file->data-store-orig
  "Returns a seq of maps containing features read from the shape file.
   kind is either gadm or natural-earth. file is anything that can be given to io/file"
  [file]
  (-> file io/file FileDataStoreFinder/getDataStore))

(defn shape-file->data-store
  "Returns a seq of maps containing features read from the shape file.
   kind is either gadm or natural-earth. file is anything that can be given to io/file"
  [file]
  (DataStoreFinder/getDataStore {"url" (-> file io/file .toURL)
                                 "memory mapped buffer" true})
  ;(-> file io/file FileDataStoreFinder/getDataStore)
  )

(defprotocol IGeo
  (read-feature [this feature])
  (locate [this lat lng]))

(defrecord NaturalEarthShapeFileGeo [file]
  component/Lifecycle
  (start [this]
    (log/infof "NaturalEarthShapeFileGeo initializing with file: %s" file)
    (assoc this :data-store (shape-file->data-store file)))
  (stop [this]
    (when-let [^DataStore ds (:data-store this)]
      (.dispose ds))
    (dissoc this :data-store))

  IGeo
  (read-feature [this feature]
    (let [f #(.getAttribute ^SimpleFeature feature ^String %)]
      {:country (f "name")
       :iso-3 (f "iso_a3")
       :region nil
       :sub-region nil
       :sub-sub-region nil}))

  (locate [this lat lng]
    (map (partial read-feature this) (locate* (:data-store this) lat lng))))

(defrecord GadmShapeFileGeo [file]
  component/Lifecycle
  (start [this]
    (log/infof "GadmShapeFileGeo initializing with file: %s" file)
    (assoc this :data-store (shape-file->data-store file)))
  (stop [this]
    (when-let [^DataStore ds (:data-store this)]
      (.dispose ds))
    (dissoc this :data-store))

  IGeo
  (read-feature [this feature]
    (let [f #(.getAttribute ^SimpleFeature feature ^String %)]
      {:country (f "NAME_0")
       :iso-3 (f "ISO")
       :region (f "NAME_1")
       :sub-region (f "NAME_2")
       :sub-sub-region (f "NAME_3")}))

  (locate [this lat lng]
    (map (partial read-feature this) (locate* (:data-store this) lat lng))))

(defrecord MarinerRegionsShapeFileGeo [file]
  component/Lifecycle
  (start [this]
    (log/infof "MarinerRegionsShapeFileGeo initializing with file: %s" file)
    (assoc this :data-store (shape-file->data-store file)))
  (stop [this]
    (when-let [^DataStore ds (:data-store this)]
      (.dispose ds))
    (dissoc this :data-store))

  IGeo
  (read-feature [this feature]
    (let [f #(.getAttribute ^SimpleFeature feature ^String %)]
      {:country (f "Country")
       :iso-3 (f "ISO_3digit")
       :region nil
       :sub-region nil
       :sub-sub-region nil}))
  (locate [this lat lng]
    (map (partial read-feature this) (locate* (:data-store this) lat lng))))

(defn new-natural-earth-geo
  "Creates a new component that implements IGeo using the
  Natural Earth shape file data source."
  [file]
  (map->NaturalEarthShapeFileGeo {:file file}))

(defn new-gadm-geo
  "Creates a new component that implements IGeo using the
  GADM shape file data source."
  [file]
  (map->GadmShapeFileGeo {:file file}))

(defn new-mariner-regions-geo
  "Creates a new component that implements IGeo using the
  Mariner Regions shape file data source."
  [file]
  (map->MarinerRegionsShapeFileGeo {:file file}))
