(ns e85th.backend.core.identity
  (:refer-clojure :exclude [ensure])
  (:require [clojure.spec.alpha :as s]
            [e85th.commons.ex :as ex]
            [e85th.commons.ext :as ext]
            [e85th.commons.tel :as tel]
            [e85th.commons.email :as email]
            [e85th.backend.core.db :as db]
            [clj-time.core :as t]
            [clojure.core.match :refer [match]]
            [clojure.java.jdbc :as jdbc]
            [clojure.string :as str]
            [e85th.backend.core.domain :as domain]
            [clojure.set :as set]))

(def ^:const identity-in-use-err       :identity.error/in-use)
(def ^:const duplicate-identity-err    :identity.error/duplicate-identity)
(def ^:const token-expired-err        :identity.error/token-expired)
(def ^:const token-invalid-err        :identity.error/token-invalid)

(def ^:const mobile-type "mobile")
(def ^:const email-type "email")

(def ^:const email-type-id  #uuid "4e3c5337-7d7d-4398-bfa7-c4e17bbffa21")
(def ^:const mobile-type-id #uuid "b9e260b7-72ce-4b38-9689-9f45266eff43")

(def email-identity?
  "Answers if the associative data input is an email identity"
  (ext/key= :identity-type-id email-type-id))

(def mobile-identity?
  "Answers if the associative data input is an mobile identity"
  (ext/key= :identity-type-id mobile-type-id))

(def identity-type->normalizer
  {email-type email/normalize
   mobile-type tel/normalize})

(defn normalize-identifier
  [{:keys [identity-type-id identifier] :as ident}]
  (if (str/blank? identifier)
    ident
    (let [f (get identity-type->normalizer identity-type-id identity)]
      (update ident :identifier f))))

;;----------------------------------------------------------------------
(s/fdef get-by-id
        :args (s/cat :res map? :id ::domain/id)
        :ret  (s/nilable ::domain/identity))

(defn get-by-id
  [{:keys [db]} id]
  (db/select-identity-by-id db id))

(def get-by-id! (ex/wrap-not-found get-by-id))

;;----------------------------------------------------------------------
(s/fdef get-by-user-id
        :args (s/cat :res map? :user-id ::domain/user-id :identifier-type-id (s/? ::domain/id))
        :ret  (s/coll-of ::domain/identity))

(defn get-by-user-id
  "Enumerates all identities by the user-id."
  ([{:keys [db]} user-id]
   (db/select-identities-by-user-id db user-id))
  ([res user-id identifier-type-id]
   (->> (get-by-user-id res user-id)
        (filter (ext/key= :identity-type-id identifier-type-id)))))

;;----------------------------------------------------------------------
(s/fdef get-by-identifier
        :args (s/cat :res map? :identifier ::domain/identifier)
        :ret  (s/coll-of ::domain/identity))

(defn get-by-identifier
  "Enumerate all identities matching the identifier."
  [{:keys [db]} identifier]
  (db/select-identities-by-identifier db identifier))


;;----------------------------------------------------------------------
(s/fdef get-by-type
        :args (s/cat :res map? :identity-type-id ::domain/id :identifier ::domain/identifier)
        :ret  (s/nilable ::domain/identity))

(defn get-by-type
  [{:keys [db]} identity-type-id identifier]
  (db/select-identity-by-type db identity-type-id identifier))

(def get-by-type! (ex/wrap-not-found get-by-type))


;;----------------------------------------------------------------------
(s/fdef get-by-mobile
        :args (s/cat :res map? :mobile-nbr ::domain/identifier)
        :ret  (s/nilable ::domain/identity))

(defn get-by-mobile
  "Finds an mobile identity for the given mobile phone number."
  [res mobile-nbr]
  (get-by-type res mobile-type-id (tel/normalize mobile-nbr)))

(def get-by-mobile! (ex/wrap-not-found get-by-mobile))


;;----------------------------------------------------------------------
(s/fdef get-by-email
        :args (s/cat :res map? :email ::domain/identifier)
        :ret  (s/nilable ::domain/identity))

(defn get-by-email
  "Finds an email identity for the given email address."
  [res email]
  (get-by-type res email-type-id (email/normalize email)))

(def get-by-email! (ex/wrap-not-found get-by-email))


;;----------------------------------------------------------------------
(s/fdef create
        :args (s/cat :res map? :identity ::domain/identity :user-id ::domain/user-id)
        :ret  ::domain/id)

(defn create
  "Creates a new identity and returns the Identity record."
  [{:keys [db] :as res} identity user-id]
  (db/insert-identity db (normalize-identifier identity) user-id))

;;----------------------------------------------------------------------
(s/fdef update-by-id
        :args (s/cat :res map? :id ::domain/id :identity map? :user-id ::domain/user-id)
        :ret  int?)

(defn update-by-id
  "Update the identity attributes and return the updated Identity record."
  [{:keys [db] :as res} id identity user-id]
  (db/update-identity-by-id db id (normalize-identifier identity) user-id))

;;----------------------------------------------------------------------
(s/fdef delete-by-id
        :args (s/cat :res map? :id ::domain/id)
        :ret any?)

(defn delete-by-id
  [{:keys [db]} id]
  (db/delete-identity-by-id db id))

;;----------------------------------------------------------------------
(s/fdef ensure
        :args (s/cat :res map? :new-ident ::domain/identity :modifier-user-id ::domain/user-id)
        :ret (s/tuple keyword? ::domain/identity))


(defn ensure
  "Ensures the identity exists for the user. Returns a variant [status ident]
   Status can be :ok if already exists,
   or ::identity-in-use if the identity already belongs to another user
   or :created when the new identity is created."
  [res {:keys [identity-type-id identifier user-id] :as new-ident} modifier-user-id]
  (if-let [ident (get-by-type res identity-type-id identifier)]
    (if (= user-id (:user-id ident))
      [:ok ident]
      [identity-in-use-err ident])
    [:created (create res new-ident modifier-user-id)]))

;;----------------------------------------------------------------------
(s/fdef used-by-another?
        :args (s/cat :res map? :identity-type-id ::domain/id :identifier ::domain/identifier :user-id ::domain/user-id)
        :ret boolean?)

(defn used-by-another?
  "Answers if identity is being used by another user."
  [res identity-type-id identifier user-id]
  (if-let [ident (get-by-type res identity-type-id identifier)]
    (not= (:user-id ident) user-id)
    false))

;;----------------------------------------------------------------------
(s/fdef persist
        :args (s/cat :res map? :new-data map? :clear-verified? boolean? :modifier-user-id ::domain/user-id)
        :ret (s/tuple keyword? any?))

(defn- persist
  "Either a variant [status identity] Status can be one of :no-action :created :removed :updated or identity-in-use-err
   identity can be nil when no-action is required in the degenerate case of persisting blank identifier
   when no identity exists for that identity-type.
   Expects there to be at most 1 identity for the type otherwise throws an exception."
  [{:keys [db] :as res} {:keys [user-id identifier identity-type-id] :as new-data} clear-verified? modifier-user-id]
  (if (and (seq identifier)
           (used-by-another? res identity-type-id identifier user-id))
    [identity-in-use-err nil]
    (let [idents (get-by-user-id res user-id identity-type-id)
          ident (select-keys  (first idents) [:id :user-id :identity-type-id :identifier :token :token-expiration :verified-at]) ;; FIXME: this should look at all identities not just the first one
          new-identifier (:identifier new-data)
          data (cond-> new-data
                 clear-verified? (assoc :verified-at nil))]

      (match [(count idents)
              (if (str/blank? new-identifier) :new-blank :new-some)
              (if (= new-identifier (:identifier ident)) :unidentged :identged)]
             [0 :new-blank _] [:no-action nil]
             [0 :new-some _] [:created (create res (-> new-data
                                                       (assoc :id (ext/random-uuid))
                                                       (dissoc :verified-at))
                                                   modifier-user-id)]
             [1 :new-blank _] [:removed (delete-by-id res (:id ident))]
             [1 :new-some :identged] [:updated (update-by-id res (:id ident) (merge ident new-data) modifier-user-id)]
             [1 :new-some :unidentged] [:no-action ident]
             :else (throw (ex-info "Expected at most 1 identity." {:data new-data
                                                                  :count (count idents)}))))))

;;----------------------------------------------------------------------
(s/fdef persist-email
        :args (s/cat :res map? :user-id ::domain/user-id :new-email (s/nilable string?)
                     :clear-verified? boolean? :modifier-user-id ::domain/user-id)
        :ret (s/tuple keyword? any?))

(defn persist-email
  "Updates in place the email associated with the user only if it differs.
   Expects there to only be one email identity otherwise throws an exception."
  [res user-id new-email clear-verified? modifier-user-id]
  (let [data {:user-id user-id
              :identifier (some-> new-email email/normalize)
              :identity-type-id email-type-id}]
    (persist res data clear-verified? modifier-user-id)))

;;----------------------------------------------------------------------
(s/fdef persist-mobile
        :args (s/cat :res map? :user-id ::domain/user-id :new-mobile (s/nilable string?)
                     :clear-verified? boolean? :modifier-user-id ::domain/user-id)
        :ret (s/tuple keyword? any?))

(defn persist-mobile
  "Updates in place the email associated with the user only if it differs.
   Expects there to only be one email identity otherwise throws an exception."
  [res user-id new-mobile clear-verified? modifier-user-id]
  (let [data {:user-id user-id
              :identifier (some-> new-mobile tel/normalize)
              :identity-type-id mobile-type-id}]
    (persist res data clear-verified? modifier-user-id)))

;;----------------------------------------------------------------------
(s/fdef filter-existing-identifiers
        :args (s/cat :res map? :identities (s/coll-of ::domain/identity-identifier))
        :ret (s/coll-of ::domain/identity-identifier))

(defn filter-existing-identifiers
  "Returns a subset of identifiers which already exist."
  [res identities]
  (let [ident-exists? (fn [{:keys [identity-type-id identifier]}]
                       (some? (get-by-type res identity-type-id identifier)))]
    (doall (filter ident-exists? identities))))


;;----------------------------------------------------------------------
(s/fdef validate-identity-identifiers
        :args (s/cat :res map? :identities ::domain/identity-identifiers)
        :ret nil?)

(defn validate-identity-identifiers
  "Validates that the identity identifier are not already present.
   Throws a validation exception when an identifier exists."
  [res identities]
  (when-let [{:keys [identifier]} (first (filter-existing-identifiers res identities))]
    (throw (ex/validation duplicate-identity-err (format "Identifier %s already exists." identifier)))))

;;----------------------------------------------------------------------
(s/fdef get-email-identities-by-user-id
        :args (s/cat :res map? :user-id ::domain/user-id)
        :ret (s/coll-of ::domain/identity))

(defn get-email-identities-by-user-id
  "Enumerates all email identities for the user."
  [{:keys [db]} user-id]
  (filter email-identity? (db/select-identities-by-user-id db user-id)))

;;----------------------------------------------------------------------
(s/fdef verify
        :args (s/cat :res map? :token ::domain/token)
        :ret any?)

(defn verify
  "Verify the identity if the token is valid."
  [{:keys [db] :as res} token]
  (let [{:keys [id user-id token-expiration verified-at] :as ident} (db/select-identity-by-token db token)
        err (cond
              (not token-expiration) token-invalid-err
              (t/before? token-expiration (t/now)) token-expired-err)]

    (when err
      (throw (ex/validation err)))

    (let [ident-data (cond-> {:token nil :token-expiration nil}
                      (not verified-at) (assoc :verified-at (t/now)))]
      (update-by-id res id ident-data user-id))))
