(ns e85th.backend.components
  (:require [com.stuartsierra.component :as component]
            [clojure.spec.alpha :as s]
            [io.pedestal.http :as http]
            [clojure.tools.nrepl.server :as nrepl]
            [taoensso.timbre :as log]))


(s/def ::port nat-int?)
(s/def ::host string?)
(s/def ::bind string?)

(s/def ::repl-server-opts (s/keys :req-un [::port]
                                  :opt-un [::bind]))

(defrecord Pedestal [service-map app make-routes]
  component/Lifecycle
  (start [this]
    (assert app "The app dependency has not been set. Does your system have an :app key?")
    (log/infof "Starting Pedestal Service.")
    (if (:service this)
      this
      (assoc this :service
             (-> service-map
                 (assoc ::http/routes (make-routes app))
                 http/create-server
                 http/start))))
  (stop [this]
    (log/infof "Stopping Pedestal Service.")
    (some-> this :service http/stop)
    (dissoc this :service)))

(defn new-pedestal
  [service-map make-routes]
  (map->Pedestal {:service-map service-map
                  :make-routes make-routes}))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; nRepl server
(defrecord ReplServer [bind port]
  component/Lifecycle
  (start [this]
    (if (and bind port)
      (do
        (log/infof "Starting nRepl server on %s:%s" bind port)
        (assoc this :server (nrepl/start-server :bind bind :port port)))
      (do
        (log/info "Skipping nRepl server initialization.  Both bind and port must be specified.")
        this)))

  (stop [this]
    (log/infof "Stopping nRepl server.")
    (some-> this :server nrepl/stop-server)
    (assoc this :server nil)))

;;----------------------------------------------------------------------
(s/fdef new-repl-server
        :args (s/cat :repl-opts ::repl-server-opts)
        :ret any?)

(defn new-repl-server
  "Create a new nrepl server running on host and port"
  [repl-opts]
  (let [params (merge {:bind "0.0.0.0"} repl-opts)]
    (map->ReplServer params)))
