(ns e85th.backend.core.address
  (:require [e85th.backend.core.db :as db]
            [e85th.backend.core.models :as m]
            [e85th.commons.ex :as ex]
            [taoensso.timbre :as log]
            [clojure.java.jdbc :as jdbc]
            [clojure.spec.alpha :as s]
            [e85th.backend.core.domain :as domain]
            [clojure.string :as str]))

;;----------------------------------------------------------------------
(s/fdef get-by-id
        :args (s/cat :res map? :id ::domain/id)
        :ret  (s/nilable ::domain/address))

(defn get-by-id
  "Gets an address by its id."
  [{:keys [db]} id]
  (db/select-address-by-id db id))

(def get-by-id! (ex/wrap-not-found get-by-id))

;;----------------------------------------------------------------------
(s/fdef get-by-public-id
        :args (s/cat :res map? :id (s/coll-of ::domain/public-id))
        :ret  (s/nilable ::domain/address))

(defn get-by-public-id
  "Gets address by public id."
  [{:keys [db] :as res} id]
  (db/select-address-by-public-id db id))

(def get-by-public-id! (ex/wrap-not-found get-by-public-id))

;;----------------------------------------------------------------------
(s/fdef create
        :args (s/cat :res map? :address ::domain/address :user-id ::domain/user-id)
        :ret  (s/nilable ::domain/id))

(defn create
  "Create a new address record. Returns the address ID."
  [{:keys [db] :as res} address user-id]
  (db/insert-address db address user-id))

;;----------------------------------------------------------------------
(s/fdef update-by-id
        :args (s/cat :res map? :id ::domain/id :address ::domain/address :user-id ::domain/user-id)
        :ret  int?)

(defn update-by-id
  "Updates the address by the db id and returns the count of rows updated."
  [{:keys [db] :as res} id address user-id]
  (db/update-address-by-id db id address user-id))

;;----------------------------------------------------------------------
(s/fdef update-by-public-id
        :args (s/cat :res map? :public-id ::domain/public-id :address ::domain/address :user-id ::domain/user-id)
        :ret  int?)

(defn update-by-public-id
  "Updates the address by the public id and returns the count of rows updated."
  [{:keys [db] :as res} public-id address user-id]
  (db/update-address-by-public-id db public-id address user-id))

;;----------------------------------------------------------------------
(s/fdef same?
        :args (s/cat :address-1 ::domain/address :address-2 ::domain/address)
        :ret  boolean?)

(defn same?
  "Address equality based on street-1, street-2, city, state and postal-code."
  [address-1 address-2]
  (let [norm #(str/lower-case (str/trim (or % "")))
        str-eq? #(= (norm %1) (norm %2))
        prop-eq? (fn [k]
                   (str-eq? (k address-1) (k address-2)))]
    (every? prop-eq? [:street-1 :street-2 :city :state :postal-code])))


;;----------------------------------------------------------------------
(s/def ::identifier (s/or :id ::domain/id
                          :public-id ::domain/public-id))
(s/fdef save*
        :args (s/cat :f fn? :res map? :address-id (s/nilable ::identifier) :new-address ::domain/address :user-id ::domain/user-id)
        :ret  ::domain/id)

(defn- save*
  "Creates a new address and returns the address id when address-id is nil or
   new address is different from the address referenced by address-id based on same?.
   Otherwise returns the current address id."
  [f res address-id new-address user-id]
  (let [db-address (some->> address-id (f res))
        db-id (:id db-address)
        address-same? (and db-address (same? db-address new-address))]

    (if address-same?
      db-id
      (:id (create res new-address user-id)))))


(def save-by-id (partial save* get-by-id))
(def save-by-public-id (partial save* get-by-public-id))


(s/fdef assoc-user
        :args (s/cat :res map? :address ::domain/address :user-id ::domain/user-id :creator-id ::domain/user-id))

(defn assoc-user
  "Create an address and assoc it to the user."
  [{:keys [db] :as res} address user-id creator-id]
  (let [address-id (:id (create res address creator-id))]
    (db/insert-user-address db user-id address-id creator-id)))
