(ns onto.core
  (:require [clojure.test.check]
            [clojure.spec :as s]
            [clojure.spec.gen :as gen]
            [clojure.spec.test :as stest])
  (:import clojure.test.check.generators.Generator)
  (:gen-class))




;; Standard symbols

(def |8

     "Alias for Infinity
      Ex.  (|gen ::.Int |8)   => Infinite lazy seq of Ints"
     Double/POSITIVE_INFINITY)


(def |x

     "Alias for :clojure.spec/invalid"
     ::s/invalid)


(def |= `|=)
(def |< `|<)
(def |> `|>)
(def |- `|-)
(def _! `_!)
(def !! `!!)
(def _? `_?)
(def ?? `??)


(def |$

     "Alias for clojure.spec/registry"
     s/registry)


(def G-nil

     "A generator always returning nothing"
     (gen/return nil))






;; Spec definition


(defmacro -|

  "Define a new spec

   |-  Spec definition is...
   |<  Based on spec...
   |>  Create a generator using fn

   |-                 is mandatory
   |<  without  |>    generator will be that of the base spec
   |>  without  |<    generator will pull values from no op fn
   |<  with     |>    generator will map base spec generator to fn

   Definition of a String
   (-|  ::.Str
     |- string?)

   Definition of a String starting with \"Foo\" based on ::.Str, generator included
   (-|  ::.Foo
     |- (|& ::.Str
            #(re-find #\"^Foo\" %))
     |< ::.Str
     |> #(str \"Foo\" %))
   
   Definition of an Int between 1 and 10, standalone generator include
   (-|  ::.1<=.Int.<10
     |- (|& int?
            #(<= 1 %)
            #(<= % 10))
     |> #(+ (rand-int 10)
            1))

   Figure this one
   (-|  ::.Universe
     |- #(= % 42)
     |> (constantly 42))

   
   Onto Core support spec'ing for anonymous (not namespaced) kws by wrapping them in a random one. 
    
   (-|  :foo
     |- ::.Str)     =>  :onto.random.13045/foo
  
   Very useful for defining maps with anonymous (not namespaced) keys, specially if different maps uses
   same keys for different meanings.
   For more refer to |k

   Anonymous syms will be namespaced with the defining namespace."
   

  [sym|kw & {:syms [|-
                    |<
                    |>]}]
  (let [sym|kw* (if (or (symbol?   sym|kw)
                        (namespace sym|kw))
                    sym|kw
                    (keyword (str (gensym "onto.random.")
                                  "/"
                                  (name sym|kw))))]
    `(s/def ~sym|kw*
            ~(if |> `(s/with-gen ~|-
                                 (fn [] (gen/fmap ~@(if |< `[~|>
                                                             (s/gen ~|<)]
                                                           `[(fn [_#] (~|>))
                                                             G-nil]))))
                    |-))))





;; Spec'ing a fn


(defmacro F|

  "Alias for clojure.spec/fdef

   |<  :args
   |>  :ret
   |=  :fn

   Fn foo take an predefined Int N and returns a Coll of Ints such that Coll is size N
   (F|  foo
     |< (r- :N ::.Int)
     |> (|c ::.Int)
     |= #(= (count (|>? %))
            (:N (|<? %))))"

  [sym & {:syms [|<
                 |>
                 |=]
          :keys [args
                 ret
                 fn]}]
  `(s/fdef ~sym
           :args ~(or |< args)
           :ret  ~(or |> ret)
           :fn   ~(or |= fn)))



(def |<?

     "Helper for F|
      Alias for :args allowing expressiveness in an F| body"
     :args)


(def |>?

     "Helper for F|
      Alias for :ret allowing expressiveness in an F| body"
     :ret)







(defmacro |not

  "Negate a spec relatively to a superset
   

   Definition of non-0 negative numbers using the definition of positive numbers

   (-|  ::.Num.+
     |- (|& ::.Num
            #(>= % 0)))

   (-|  ::.Num.*-
     |-  (|& ::.Num
             (|not ::.Num.+)))

   Definition of something that is not a number
   ::.Any must be provided so the engine has a generator to start with, hence the idea of a superset
   
   (-|  ::<>Num
     |- (|& ::.Any
            (|not ::.Num)))

   This alternative definition understand what is not a number but is unable to generate values
   
   (-|  ::<>Num
     |- (|not ::.Num))"

  [spec]
  `(fn [?#] (not (s/valid? ~spec ?#))))






;; Spec combinators and data structures definition

(defmacro |&

          "Alias for clojure.spec/and"
          [& body]
          `(s/and ~@body))


(defmacro ||

          "Alias for clojure.spec/or"
          [& body]
          `(s/or ~@body))




(defmacro |?

          "Alias for clojure.spec/nilable"
          [& body]
          `(s/nilable ~@body))




(defmacro |c

          "Alias for clojure.spec/coll-of"
          [& body]
          `(s/coll-of ~@body))


(defmacro |t

          "Alias for clojure.spec/tuple"
          [& body]
          `(s/tuple ~@body))


(defmacro |m

          "Alias for clojure.spec/map-of"
          [& body]
          `(s/map-of ~@body))






(defn- -|?
      
      "Guess if ? is a spec definition"
      [?] (true? (and (list? ?)
                           (= (first ?) '-|))))



(defn- keys-sub

  "Convert Onto symbols to proper keywords used by clojure.spec/keys and clojure.spec/keys*
   but still supports those keywords.

   Wrap values given to those 'modifier' syms in a vector if needed.
  
   If a -| definition is spotted, it is evaled.
   This allow to declare specs such as (-| :foo |- ::.Int) within  |k  or  |k*"

  [{:syms [_! !! _? ??]
    :keys [req-un req
           opt-un opt]}]
  (reduce-kv (fn [acc k v]
               (if v (-> acc
                         (conj k)
                         (conj (if (-|? v) [(eval v)]
                                           (map (fn [?] (if (-|? ?) (eval ?)
                                                                    ?))
                                                (if (coll? v) v
                                                              [v])))))
                     acc))
             []
             {:req-un (or _! req-un)
              :req (or !! req)
              :opt-un (or _? opt-un)
              :opt (or ?? opt) }))


(defmacro |k
  
  "Alias for clojure.spec/keys with kw substitutes for expressiveness and concision
   
   _!  :req-un
   !!  :req
   _?  :opt-un
   ??  :opt

   Those syms are modifiers, in fact
  

   Ex. Foo-Bar has anonymous (non-namespaced) keys :foo and :bar, and maybe namespaced :baz

   (-|  ::.Foo-Bar
     |- (|k _! [::foo
                ::bar]
            ?? [::baz]))

   If a single spec is provided to a modifier, [] in optional around this spec
   
   (|k _! ::foo)  ==  (|k _! [::foo])


   Often, different kind of maps within a namespace use same kws for different meaning
   This is a common anti-pattern to avoid clashes :

   (|k _? [::phone
           :some.useless.namespace/name])
   
   (-| :some.useless.namespace/name
     |- ::.Str)
  
   some.useless.namespace doesn't provide semantics, it is merely a placeholder and
   can result in namespace collisions
   Onto Core provides spec'ing of anonymous kws|syms and suggests this approach :

   (|k _? [::phone
           (-| :name |- ::.Str)])
    
   :name is wrapped with a randomly generated namespace and defined as a ::.Str
   Refer to -| for more about spec'ing anonymous syms|kws"

  [& body]
  `(s/keys ~@(keys-sub body)))



(defmacro |k*

          "Alias for clojure.spec/keys*
           See  |k  for semantics"
          [& body]
          `(s/keys* ~@(keys-sub body)))



(defmacro |M

          "Alias for clojure.spec/merge"
          [& body]
          `(s/merge ~@body))








;; Helpers for regex

(defmacro r*

          "Alias for clojure.spec/*"
          [?]
          `(s/* ~?))


(defmacro r+

          "Alias for clojure.spec/+"
          [?]
          `(s/+ ~?))


(defmacro r?

          "Alias for clojure.spec/?"
          [?]
          `(s/? ~?))


(defmacro r&

          "Alias for clojure.spec/&"
          [& body]
          `(s/& ~@body))


(defmacro r-

          "Alias for clojure.spec/cat"
          [& body]
          `(s/cat ~@body))


(defmacro r|

          "Alias for clojure.spec/alt"
          [& body]
          `(s/alt ~@body))









;; Helpers for clojure.spec.test

(def |in

     "Alias for clojure.spec.test/instrument"
     stest/instrument)


(def |un

     "Alias for clojure.spec.test/unstrument"
     stest/unstrument)

(def |ck

     "Alias for clojure.spec.test/check"
     stest/check)







;; Misc

(defn |o

  "Given a Class/Type, returns a fn testing a the membership of a value against it"
  [cl] #(instance? cl %))





;; Test values against specs

(def |vl?

     "Alias for clojure.spec/valid?"
     s/valid?)

(def |exp

     "Alias for clojure.spec/explain"
     s/explain)








;; Generate values following specs

(defn get-gen [S|G] (if (instance? Generator S|G) S|G
                                                  (s/gen S|G)))

(defn |gen
      
  "Generate 1 or N values from a Spec or a Generator"

  ([S|G]
   (gen/generate (get-gen S|G)))
  ([S|G n]
   (gen/sample (get-gen S|G) n)))


(def |trS

     "Alias for clojure.spec/exercise
      Train a Spec"
     s/exercise)

(def |trF
     
     "Alias for clojure.spec/exercise-fn
      Train a speced fn"
     s/exercise-fn)







(defn |try

  "Try all specs in a namespace n times.
   Prints all keywords/symbols in this ns one by one and throw if something goes wrong.
   Default n = 1000
  
  (|try \"onto.std\")"
  

  ([namesp] (|try namesp 1000))

  ([namesp n]
   (println \newline
            "Testing specs in" namesp
            \newline)
   (let [kws (->> (|$)
                  (filter (fn [[k]] (= (namespace k) namesp)))
                  (map (fn [[k]] k)))
         c (count kws)]
     (println "Found" c "specs"
              \newline)
     (doseq [kw kws]
       (println kw)
       (try (|gen kw n)))
     (println \newline
              (if (= c 0)
                  "Nothing to try"
                  "Great ! Nothing bad to report about value generation, but check if they make sense")))))
