(ns dvlopt.flexo

  "............................................................................
   .                                                                          .
   .  - A context is a flat map containing data and functions handling data.  .
   .                                                                          .
   .  - A handler - is a function accepting a context map and a value.        .
   .                                                                          .
   .              - returns - a new value                                     .
   .                        - produces a side effect                          .
   .                        - or both                                         .
   .                                                                          .
   ............................................................................"

  {:author "Adam Helinski"})




;;;;;;;;;; Helpers for handlers


(defn nothing

  "Handler that does absolutely nothing."

  [_ctx _value]

  nil)




(defn constant

  "Makes a handler always returning the same value."

  ([]

   nothing)


  ([value]

   (fn return-constant [_ctx _value]
     value)))




;;;;;;;;;; Handle and signal values


(defn handle

  "Given a `ctx`, gets the handler using `k` and handles the value `v` (nil if not given).

   If the handler is not found, returns `v` or `v-default` (nil if given)."

  ([ctx k]

   (handle ctx
           k
           nil
           nil))


  ([ctx k v]

   (handle ctx
           k
           v
           v))


  ([ctx k v v-default]

   (if-let [handler (get ctx
                         k)]
     (handler ctx
              v)
     v-default)))




(defn handle-any

  "Similar to `handle`.

   Given a list of `ks`, use the first found handler."

  ([ctx ks]

   (handle-any ctx
               ks
               nil
               nil))


  ([ctx ks v]

   (handle-any ctx
               ks
               v
               v))


  ([ctx ks v v-default]

   (if (seq ks)
     (if-let [handler (get ctx
                           (first ks))]
       (handler ctx
                v)
       (recur ctx
              (rest ks)
              v
              v-default))
     v-default)))




(defn signal

  "Behaves like `handle` but used for side effects.
  
   Always returns `v`."

  ([ctx k]

   (signal ctx
           k
           nil))


  ([ctx k v]

   (handle ctx
           k
           v
           nil)
   v))




(defn signal-any

  "Behaves like `handle-any` but used for side effects.
  
   Always returns `v`."

  ([ctx ks]

   (signal-any ctx
               ks
               nil))


  ([ctx ks v]

   (handle-any ctx
               ks
               v
               nil)
   v))




 (defn signal-all

   "Similar to `signal-any`, will call every found handler."

   ([ctx ks]

    (signal-all ctx
                ks
                nil))


   ([ctx ks v]

    (doseq [k ks]
      (handle ctx
              k
              v))
    v))
