(ns dosql.ring.service
  (:require [clojure.tools.logging :as log]
            [clojure.set]
            [clojure.string]
            [ring.middleware.params :as p]
            [ring.middleware.nested-params :as np]
            [ring.middleware.keyword-params :as kp]
            [ring.middleware.format-params :as fp]
            [ring.middleware.format-response :as fr]
            [dosql.ring.util :as m]
            [dosql.clj.fail :as f]
            [dosql.jdbc-io :as io]
            [dosql.core :as tj]))



(defn- http-response [v]
  {:status  200
   :headers {"Content-Type" "text/html; charset=utf-8"}
   :body    v})


(defn ok-response [v]
  (http-response [v nil]))


(defn error-response [e]
  (http-response [nil e]))


(defn response
  [m]
  (if (f/failed? m)
    (do
      (log/info "failed response " (into {} m))
      (error-response (str (into {} m))))
    (ok-response m)))



(defn is-valid?
  [params]
  (if (and params
           (not-empty params))
    params
    (f/fail "Param is empty ")))




(def do-params (-> m/process-params
                   (kp/wrap-keyword-params)
                   (np/wrap-nested-params)
                   (p/wrap-params)
                   (fp/wrap-restful-params)))


(defn get-ds-name
  ([params ring-request config]
   (or
     (:dosql.core/ds params)
     (get-in ring-request [:session :dosql.core/ds])
     (:ds-name config)
     "default"))
  ([config]
   (or (:ds-name config)
       "default")))



(defn dispatch-type [{:keys [pull-path push-path log? encoding]
                      :or   {pull-path "/pull"
                             push-path "/push"}} req]
  (let [request-path (clojure.string/split (or (:path-info req)
                                               (:uri req)) #"/")
        op-name (str "/" (last request-path))]
    (condp = op-name
      pull-path :pull
      push-path :push
      :default)))


(defmulti apply-api (fn [m _ _ req] (dispatch-type m req)))


(defmethod apply-api :default
  [_ _ _ _]
  (f/fail " API not found "))


(defmethod apply-api :pull
  [{:keys [encoding]} ds-m config req]
  (let [params (do-params req)
        tms (:tms config)
        ds-name (get-ds-name params req config)
        ds (get ds-m ds-name)
        params (dissoc params :dosql.core/ds)]
    (log/info "Processing params with ds " params ds-name)
    (f/try->> params
              (is-valid?)
              (tj/pull ds tms)
              (m/format-response req))))


(defmethod apply-api :push
  [{:keys [encoding]} ds-m config req]
  (let [params (do-params req)
        tms (:tms config)
        ds-name (get-ds-name params req config)
        ds (get ds-m ds-name)]
    (log/info "Processing params with ds " params ds-name)
    (f/try->> params
              (is-valid?)
              (tj/push! ds tms))))

;(fr/wrap-restful-response)


(declare load-module)


(defn reload-and-get-config [ds-m-atom config-atom req-url]
  (if-let [config (get @config-atom req-url)]
    (if (:reload? config)
      (let [r (load-module @ds-m-atom config)]
        (swap! config-atom assoc req-url r)
        r)
      config)))




(defn warp-dosql
  "Warper that tries to do with dosql. It should use next to the ring-handler. If path-in is matched with
   pull-path or push-path then it will API and return result.

   handler: Ring handler
   ds-atom: Clojure datasource as atom
   tms-atom: dosql file as atom
   pull-path and push path string

  "
  [handler ds-m-atom dosql-coll-atom & {:as m}]
  (fn [req]
    (when (:log? m)
      (log/info "Porcessing request..." (m/pformat (dissoc req :server-exchange :body))))
    (let [request-path (clojure.string/split (or (:path-info req)
                                                 (:uri req))
                                             #"/")
          req-url (clojure.string/join "/" (butlast request-path))]
      (if (get @dosql-coll-atom req-url)
        (let [config (reload-and-get-config ds-m-atom dosql-coll-atom req-url)
              h (fn [r] (-> (apply-api m @ds-m-atom config r)
                            (response)))
              h1 (-> h
                     (fr/wrap-restful-response))]
          (h1 req))
        (handler req)))))


(defn file-name->url [file-name]
  (-> (clojure.string/split file-name #"\/")
      (last)
      (clojure.string/split #"\.")
      (first)
      ))


(comment
  (file-name->url "ghello.edn.sql")

  )


(defn load-module
  [ds-m {:keys [file-name init-name url]
         :as   m}]
  (try
    (let [v (tj/read-file file-name)
          ds-name (get-ds-name m)
          ds (get ds-m ds-name)]
      (do
        (log/info "Loading file with ds  " file-name ds-name)
        (when init-name
          (log/info "Doing database init " init-name)
          (io/db-do ds (tj/select-name v {:dosql.core/name init-name})))
        (io/validate-dml! ds (tj/get-sql-statement v))
        (-> m
            (assoc :url (or url (str "/" (file-name->url file-name))))
            (assoc :tms v)
            (dissoc :init-name))))
    (catch Exception e
      (log/error e (str " File loading error " (str m)))
      nil)))


(defn load-module-batch [ds-m tms-coll]
  (->> (mapv (partial load-module ds-m) tms-coll)
       (remove nil?)
       (mapv (fn [v] {(:url v) v}))
       (into {})))
