(ns dosql.jdbc-io
  (:require [clojure.java.jdbc :as jdbc]
            [dosql.clj.interceptor :as inter]
            [io.pedestal.interceptor.helpers :as h]
            [io.pedestal.interceptor.chain :as c]
            [dosql.clj.common :as cc]
            [dosql.clj.fail :as f]
            [clojure.tools.logging :as log]))


(defn jdbc-handler2
  [tm]
  (log/debug tm)
  ;(clojure.pprint/pprint tm )
  (let [dml-type (:dosql.core/dml tm)
        ds (:dosql.core/ds tm)
        sql (:dosql.core/sql tm)
        result (:dosql.core/result tm)
        st (System/nanoTime)
        stm (System/currentTimeMillis)
        output (condp = dml-type
                 :dosql.core/dml-select
                 (if (contains? result :dosql.core/result-array)
                   (jdbc/query ds sql :as-arrays? true :identifiers clojure.string/lower-case)
                   (jdbc/query ds sql :as-arrays? false :identifiers clojure.string/lower-case))
                 :dosql.core/dml-insert
                 (jdbc/execute! ds sql :multi? true)
                 (jdbc/execute! ds sql))
        total (/ (clojure.core/double
                   (- (System/nanoTime)
                      st)) 1000000.0)
        tm (merge tm {:dosql.core/output          output
                      :dosql.core/exec-total-time total
                      :dosql.core/exec-start-time stm})]
    (log/info "SqlIO: " ((juxt :dosql.core/name :dosql.core/exec-total-time :dosql.core/sql) tm))
    tm))



#_(defn parallel-load-handler [ds tm-coll]
    (-> {:request tm-coll}
        (c/execute [inter/parallel-thread-interceptor
                    (h/handler (partial jdbc-handler2 ds))])
        (:response)))


(defn is-rollback?
  [commit-type read-only]
  (cond
    (= true read-only) true
    (= commit-type :dosql.core/commit-none) true
    :else false))


(defn commit-type
  "Return commit type if not found return commit-none-key  "
  [tm-coll]
  (let [p (comp
            (filter #(not= :dosql.core/dml-select (:dosql.core/dml %)))
            (map #(:dosql.core/commit %))
            (map #(or % :dosql.core/commit-all)))
        commits (into [] p tm-coll)]
    ;(println commits)
    (if (empty? commits)
      :dosql.core/commit-none
      (or (some #{:dosql.core/commit-none} commits)
          (some #{:dosql.core/commit-all} commits)
          (cc/contain-all? commits :dosql.core/commit-any)
          :dosql.core/commit-none))))


(defn read-only?
  "Is it read only"
  ([tx-map]
   (if (contains? tx-map :read-only?)
     (:read-only? tx-map)
     true)))


(defn global-info-m [tms m-coll]
  (let [tx-prop (apply hash-map (get-in tms [:_global_ :dosql.core/tx-prop]))
        isolation (or (:isolation tx-prop) :serializable)
        read-only? (read-only? tx-prop)
        commit-type (is-rollback? (commit-type m-coll) read-only?)]
    {:isolation  isolation
     :read-only? read-only?
     :rollback?  commit-type}))


#_(defn with-transaction-handler [ds tms tm-coll]
    (let [m (global-info-m tms tm-coll)
          {:keys [isolation read-only? rollback?]} m]
      (jdbc/with-db-transaction
        [t-conn ds
         :isolation isolation
         :read-only? read-only?]
        (let [result (-> {:request tm-coll}
                         (c/execute [inter/sequence-until-error-interceptor
                                     (h/handler (partial jdbc-handler2 t-conn))])
                         (:response)
                         #_(merge-output tm-coll))]
          (when rollback?
            (jdbc/db-set-rollback-only! t-conn))
          result))))


(defn validate-dml! [ds sql-str-coll]
  (jdbc/with-db-connection
    [conn ds]
    (doseq [str sql-str-coll]
      (jdbc/prepare-statement (:connection conn) str)))
  (log/info "Validation done for " sql-str-coll))


(defn db-do [ds sql-str-coll]
  (try
    (doseq [m sql-str-coll]
      (when-let [sql (get-in m [:dosql.core/sql])]
        (log/info "db do with " sql)
        (jdbc/db-do-commands ds sql)))
    (catch Exception e
      (do
        (log/error e)
        (f/fail {:detail e})))))
















#_(defn jdbc-handler
    [ds tm _]
    ;(println "jdbc hanbdler ")
    ;(clojure.pprint/pprint tm )
    (log/info "Sql: " (:dosql.core/sql tm))
    (let [dml-type (:dosql.core/dml tm)
          sql (:dosql.core/sql tm)
          result (:dosql.core/result tm)]
      (condp = dml-type
        :dosql.core/dml-select
        (if (contains? result :dosql.core/result-array)
          (jdbc/query ds sql :as-arrays? true :identifiers clojure.string/lower-case)
          (jdbc/query ds sql :as-arrays? false :identifiers clojure.string/lower-case))
        :dosql.core/dml-insert
        (jdbc/execute! ds sql :multi? true)
        (jdbc/execute! ds sql))))



#_(defn jdbc-handler-batch
    [ds tm-coll m]
    (let [{:keys [isolation read-only? rollback?]} m]
      (jdbc/with-db-transaction
        [t-conn ds
         :isolation isolation
         :read-only? read-only?]
        (let [result (mapv #(jdbc-handler t-conn % m) tm-coll)]
          (when rollback?
            (jdbc/db-set-rollback-only! t-conn))
          result))))
