(ns dosql.impl.sql-bind-impl
  (:require [clojure.spec :as s]
            [dosql.clj.common :as cc]))


(defn validate-input-not-empty-one!
  [sql-param param]
  (when (and (not-empty sql-param)
             (empty? param))
    (throw (ex-info (format "Input is missing  ") {}))))


(defn validate-input-not-empty!
  [m]
  (let [sql-param (rest (:dosql.core/sql m))
        [t p] (:dosql.core/param m)]
    (if (= :many t)
      (doseq [p1 p]
        (validate-input-not-empty-one! sql-param p1))
      (validate-input-not-empty-one! sql-param p))
    m))


(defn validate-input-type!
  [m]
  (let [dml-type (:dosql.core/dml m)
        [_ input] (:dosql.core/param m)
        sql (:dosql.core/sql m)]
    (if (and (not= dml-type :dosql.core/dml-insert)
             (not-empty (rest sql))
             (not (map? input)))
      (throw (ex-info (format "Input Params for %s will be map format but %s is not map format " sql input) input))
      m)))


(comment

  ;(contains? #{1 2 3}  5)

  (validate-input-type!
    {:dosql.core/sql ["insert into employee_detail (employee_id, street,   city,  state,  country )
                           values (:employee_id, :street, :city, :state, :country)"
                      :employee_id
                      :street
                      :city
                      :state
                      :country],
     :dosql.core/dml :dosql.core/dml-insert,
     :input          [{:street      "Schwan",
                       :city        "Munich",
                       :state       "Bayern",
                       :country     "Germany",
                       :id          126,
                       :employee_id 125}],
     })

  )





(defn- validate-required-params*!
  [p-set input]
  (let [p-set (into #{} p-set)
        i-set (into #{} (keys input))
        diff-keys (clojure.set/difference p-set i-set)]
    (if-not (empty? diff-keys)
      (throw (ex-info (format "Missing required parameter %s" (pr-str diff-keys)) input))
      input)))


(defn validate-required-params!
  [m]
  (let [[t v] (:dosql.core/param m)]
    (doseq [input (cc/as-sequential v)]
      (validate-required-params*! (rest (:dosql.core/sql m)) input))
    m
    )
  )


(defn is-coll? [v]
  (let [w (s/form (eval v))]
    ;(println "----" (name (first w)) )
    (if (and (coll? w)
             (= (name (first w)) "coll-of"))
      true
      false)))


(defn get-place-holder
  [type v]
  ;(println type )
  (if type
    (if (and (sequential? v)
             (is-coll? type))
      (clojure.string/join ", " (repeat (count v) "?"))
      "?")
    "?"
    )

  )


(defn replace-sql-param
  [v sql-str k]
  (clojure.string/replace-first sql-str (re-pattern (str k)) v))


(defmulti sql-bind (fn [tm] (:dosql.core/dml tm)))


(defmethod sql-bind
  :default
  [tm]
  (let [[sql-str & sql-params] (:dosql.core/sql tm)
        [_ input] (:dosql.core/param tm)
        param-spec (or (:dosql.core/param-spec tm) {})
        param-spec (merge (:opt param-spec) (:req param-spec))
        rf (fn [sql-coll p-key]
             (let [p-value (cc/as-sequential (p-key input))
                   w (-> (p-key param-spec)
                         (get-place-holder p-value)
                         (replace-sql-param (first sql-coll) p-key))
                   q-str (assoc sql-coll 0 w)]
               (reduce conj q-str p-value)))]
    ; (clojure.pprint/pprint  (reduce rf [sql-str] sql-params))
    (->> (reduce rf [sql-str] sql-params)
         (assoc tm :dosql.core/sql))))


(defmethod sql-bind
  :dosql.core/dml-insert
  [tm]
  (let [sql (:dosql.core/sql tm)
        sql-str (reduce (partial replace-sql-param "?") sql)
        [t param] (:dosql.core/param tm)
        ]
    (->> param
         (cc/as-sequential)
         (mapv #(cc/select-values %1 (rest sql)))
         (reduce conj [sql-str])
         (assoc tm :dosql.core/sql))))



