
(ns dobby.tool.render
  (:import [java.text SimpleDateFormat]
           [java.util TimeZone])
  (:refer-clojure :exclude [format])
  (:require [clojure.java.io       :as io]
            [clojure.pprint        :refer [pprint]]
            [clojure.tools.logging :as log]
            [familiar.core         :as f :refer [fmtstr format]]))

(defprotocol ^:private DepfileFormatter
  (format-header  [this])
  (format-comment [this line])
  (format-pkgspec [this pkgspec])
  (format-trailer [this]))

(defmulti ^:private make-formatter identity)

(defmethod ^:private make-formatter ::edn
  [_]
  (reify
    DepfileFormatter
    (format-header  [this] (format-comment this (fmtstr "-*- mode: clojure -*-~%")))
    (format-comment [_ comment] (fmtstr ";; ~a" comment))
    (format-pkgspec [_ pkgspec] (with-out-str (pprint pkgspec)))
    (format-trailer [_] nil)))

(defn ^:private make-banner
  [now]
  [(fmtstr "Generated by Dobby at ~a." now)
   "Do not edit."
   "Do not not put under version control."
   "Apply changes to .dobby/dep.edn."])

(defn render
  [pkgspec f target]
  (let [formatter (make-formatter target)
        fbanner   (map #(format-comment formatter %)
                       (make-banner (.format (doto (SimpleDateFormat. "yyyy-MM-dd'T'HH:mm:ss.SSS 'UTC'")
                                               (.setTimeZone (TimeZone/getTimeZone "UTC")))
                                             (java.util.Date.))))]
    (with-open [w (io/writer f)]
      (when-let [header (format-header formatter)]
        (format w "~a~%" header))
      (format w "~{~a~^~%~}~%~%" fbanner)
      (format w "~a~%" (format-pkgspec formatter pkgspec))
      (when-let [trailer (format-trailer formatter)]
        (format w "~a~%" trailer)))))
