
(ns dobby.tool.cli
  (:import  [java.nio.file Files]
            [java.util Arrays])
  (:require [clojure.data.json           :as json]
            [clojure.edn                 :as edn]
            [clojure.java.io             :as io]
            [clojure.string              :as str]
            [clojure.tools.logging       :as log]
            [clojure.walk                :as walk]

            [cognitect.aws.client.api    :as aws]
            [com.stuartsierra.dependency :as dep]
            [familiar.core               :as f :refer [fmtstr]]
            [martian.core                :as martian]
            [martian.clj-http            :as martian.http]
            [schema.core                 :as schema]

            [dobby.tool.core             :as core]
            [dobby.tool.render           :as render]
            [dobby.tool.render.clj-deps  :as deps-renderer])
  (:gen-class))

(def ^:private +depspec-filename+ ".dobby/dep.edn")
(def ^:private +envspec-filename+ ".dobby/env.json")
(def ^:private +pkgspec-filename+ ".dobby/pkg.edn")
(def ^:private +rendered-depspec-filename+ "Dependencies")

(defn ^:private resolve-project-file
  [dir specfile-name]
  (log/debug "Resolve Dobby package spec file %s for project %s" specfile-name dir)
  (.getAbsoluteFile (io/file dir specfile-name)))

(defn ^:private resolve-envspec-file
  "Locate the environment spec for the Dobby workspace relative to `dir`, where `dir` is a (repository) directory
  containing a Dobby dependency template."
  [dir candidates]
  (let [dob-env-file-candidate (io/file dir +envspec-filename+)]
    (log/debug "Resolve Dobby env spec: %s" dob-env-file-candidate)
    (cond (nil? dir)            ; We've run out of parent directories to inspect.
          (throw (ex-info "Dobby environment file not found" {:candidates candidates}))
          (.exists dob-env-file-candidate)
          (.getAbsoluteFile dob-env-file-candidate)
          :else
          (recur (.getParentFile dir) (conj candidates dob-env-file-candidate)))))

(defn ^:private make-depfile-name
  [profile-name]
  (if (= ::core/root profile-name)
    +rendered-depspec-filename+
    (fmtstr "~a.~a" +rendered-depspec-filename+ (name profile-name))))

(defn ^:private compile-project-specs
  [project-dir target-tool]
  (let [depspec-file (resolve-project-file project-dir +depspec-filename+)
        pkgspec-file (resolve-project-file project-dir +pkgspec-filename+)
        envspec-file (resolve-envspec-file (.getParentFile (.getCanonicalFile depspec-file)) [])
        package-name (-> pkgspec-file slurp edn/read-string :dobby.package/name)]
    ;; FIX: Validate input against a schema/spec so that later processing can be simlified by assuming correctness.
    (println "Rendering dependencies for" package-name)
    (doseq [[name pkgspec] (core/reify-deps (io/reader depspec-file) (io/reader envspec-file))]
      (render/render (-> pkgspec :dependencies (dissoc package-name))
                     (io/file project-dir (make-depfile-name name))
                     target-tool))))

(comment
  (def project-dir "/Users/marc/Development/sinistral/dobby/prove/proof-project/App")
  (compile-project-specs project-dir ::render/end)
  )

(def ^:private dobby-service-endpoints
  {"us-east-1" "https://d0y7qez5vf.execute-api.us-east-1.amazonaws.com/prod"
   "eu-central-1" "https://pw7l42ltq0.execute-api.eu-central-1.amazonaws.com/prod"})

(defn- resolve-service-endpoint
  []
  (let [dobby-region (or (System/getenv "dobbyRegion") (System/getenv "AWS_REGION"))]
    (or (get dobby-service-endpoints dobby-region)
        (throw (ex-info (format "dobby build region \"%s\" is not specified, or is not yet supported" dobby-region)
                        {:AWS_REGION        (System/getenv "AWS_REGION")
                         :dobbyRegion       (System/getenv "dobbyRegion")
                         :supported-regions (keys dobby-service-endpoints)})))))

(defn- dobby-service
  []
  (martian.http/bootstrap (resolve-service-endpoint)
                          [{:route-name  :catalogue-packages
                            :method      :get
                            :path-parts  ["/" :account-id "/catalogue/" :catalogue-id "/packages"]
                            :path-schema {:account-id   schema/Str
                                          :catalogue-id schema/Str}
                            :produces    ["application/json"]}]))

(defn- read-catalogue-packages
  [account-id catalogue-id]
  (->> (martian/response-for (dobby-service)
                            :catalogue-packages
                            {:account-id (str account-id) :catalogue-id (str catalogue-id)})
       :body))

(comment
  (def account-id "c9b97d25-fb22-47bc-8f9d-820d733b803e")
  (def catalogue-id "9df99195-811a-452d-87d9-ca4f4cbc00c9")

  (def args [nil (io/file project-dir +envspec-filename+)])

  (defn read-catalogue-packages [account-id catalogue-id]
    [{:packageGroupName   "dobby",
      :packageName            "proof.lib-a",
      :packageArtefactVersion {:major "0", :minor "lib-a-source-rev", :build "00000000-0000-0000-0000-000000000000"}}
     {:packageGroupName       "dobby",
      :packageName            "proof.lib-b",
      :packageArtefactVersion {:major "0", :minor "lib-b-source-rev", :build "00000000-0000-0000-0000-000000000000"}}
     {:packageGroupName       "dobby",
      :packageName            "proof.lib-c",
      :packageArtefactVersion {:major "1", :minor "lib-c-source-rev", :build "00000000-0000-0000-0000-000000000000"}}
     {:packageGroupName       "dobby",
      :packageName            "proof.service",
      :packageArtefactVersion {:major "0", :minor "service-source-rev", :build "00000000-0000-0000-0000-000000000000"}}]
    )

  (read-catalogue-packages "c9b97d25-fb22-47bc-8f9d-820d733b803e" "9df99195-811a-452d-87d9-ca4f4cbc00c9")

  )

(defn -main
  [& args]
  (set! *print-namespace-maps* false)
  (cond (= "env-init" (first args))
        (let [project-dir  (.getParent (io/as-file (.getAbsolutePath (io/as-file "."))))
              pkgspec      (-> (resolve-project-file project-dir +pkgspec-filename+)
                               slurp
                               edn/read-string)
              account-id   (or (System/getenv "dobbyAccountId")
                               (:dobby/account-id pkgspec)
                               (throw (ex-info "dobbyAccountId not set; an account ID is required" {})))
              catalogue-id (or (System/getenv "dobbyCatalogue")
                               (:dobby.package/initial-catalogue pkgspec)
                               (throw (ex-info "dobbyCatalogueId not set; a catalogue ID is required" {})))]
          (log/info (format "Initializing environment using account=%s, catalogue=%s"
                            account-id
                            catalogue-id))
          (binding [*out* (java.io.PrintWriter. (io/writer (io/file (second args))))]
            (json/pprint {:environment {:catalogueId       (str catalogue-id)
                                        :cataloguePackages (read-catalogue-packages account-id catalogue-id)}})))
        (= "gen-deps" (first args))
        (let [project-dir (.getParent (io/as-file (.getAbsolutePath (io/as-file "."))))]
          (compile-project-specs project-dir ::render/edn)
          (deps-renderer/render (io/as-file project-dir)
                                (reduce (fn [m f]
                                          (let [profile-name (or (-> f .getName (str/split #"\." 2) second keyword)
                                                                 ::core/root)]
                                            (assoc m profile-name (edn/read-string (slurp f)))))
                                        {}
                                        (filter (comp (partial re-matches #"Dependencies.*") #(.getName %))
                                                (Arrays/asList (.listFiles (io/as-file project-dir)))))))))
                              ; Note that we're intentionally going through the intermediate files, rather than doing
                              ; everying in memory; eating our own dog food, so to speak.
