
(ns dobby.tool.core
  "Example tasks showing various approaches."
  {:boot/export-tasks true}
  (:import  [java.nio.file Files]
            [java.io PushbackReader])
  (:require [clojure.data.json     :as json]
            [clojure.edn           :as edn]
            [clojure.string        :as str]
            [clojure.java.io       :as io]
            [familiar.core         :as f :refer [fmtstr]]
            [clojure.tools.logging :as log]))

(defn ^:private reify-pkg-version
  "Given a (version-masked) description of a `pkg` and a catalog of the particular versions of the `pkgs` (a map of name
  to version) that are available to the environment, replace the mask in the description with the actual version."
  [pkg pkgs]
  (letfn [(dobby-managed? [verspec]
            (and (map? verspec)
                 (some #(and (instance? clojure.lang.Named %)
                             (some-> % (namespace) (str/split #"\.") first (= "dobby")))
                       (keys verspec))))]
    (let [[pkg-name pkg-version] pkg
          templated-version      (:dobby.package/version pkg-version)
          catalogued-version     (get pkgs pkg-name)]
      (cond (and (dobby-managed? pkg-version) (nil? catalogued-version))
            (throw
             (ex-info (fmtstr "Dependency not found in catalogue: ~a" pkg)
                      {:package pkg :catalog pkgs}))
            (and (dobby-managed? pkg-version)
                 (not (= templated-version (select-keys catalogued-version (keys templated-version)))))
            (throw
             (ex-info (fmtstr "Dependency ~a not satisfied by catalogued version ~s; required: ~s"
                              pkg-name catalogued-version templated-version)
                      {:package            pkg
                       :catalogued-version catalogued-version
                       :templated-version  templated-version}))
            (and (dobby-managed? pkg-version))
            {pkg-name (update pkg-version :dobby.package/version merge catalogued-version)}
            :else
            {pkg-name pkg-version}))))

(defn ^:private normalise-depspec
  "Construct dependency lists from the project's dependency specification and the environment's package catalogue.

  The dependency specification is a loose description (typically omitting the build or patch number) of the packages
  that are required, while the catalogue describes the specific versions of the packages that are available for
  use. Reifying the dependency list is simply picking the specific version from the catalogue for each entry in the
  template.

  Returns a dependency spec with reified versions in a form that may treated as a a data contract by build-tool-specific
  Depfile renderers."
  [depspec catalogue]
  (letfn [(indexed-by-profile [[profile-name profile-spec]]
            [profile-name profile-spec])
          (rebase-pkgs [root-pkgs [profile-name profile-pkgs]]
            [profile-name (merge-with merge root-pkgs profile-pkgs)])
          (reify-versions [catalogue [profile-name profile-pkgs]]
            [profile-name {:dependencies (reduce #(merge %1 %2)
                                                 {}
                                                 (mapv #(reify-pkg-version % catalogue)
                                                       (:dependencies profile-pkgs)))}])]
    (let [root-pkgs (select-keys depspec [:dependencies])]
      (->> (assoc-in depspec [:profiles ::root] root-pkgs)
           (:profiles)
           (seq)
           (map indexed-by-profile)
           (map (partial rebase-pkgs root-pkgs))
           (map (partial reify-versions catalogue))
           (mapcat identity)
           (apply hash-map)))))

(defn reify-deps
  [depspec-rdr envspec-rdr]
  (letfn [(dobfile-reader [f]
            (PushbackReader. (io/reader f)))]
    (let [deps (edn/read (dobfile-reader depspec-rdr))
          pkgs (get-in (json/read envspec-rdr :key-fn keyword)
                       [:environment :cataloguePackages])]
      (normalise-depspec deps
                         (into {} (map (comp (fn [[gn n v]]
                                               [(symbol (fmtstr "~a/~a" gn n)) v])
                                             (juxt :packageGroupName :packageName :packageArtefactVersion)) pkgs))))))
(comment
  (def ^:private project-dir "/Users/marc/Development/sinistral/dobby/prove/proof-project/App")

  (set! *print-namespace-maps* false)

  (reify-deps (io/reader (io/file project-dir @#'dobby.tool.cli/+depspec-filename+))
              (io/reader (io/file project-dir @#'dobby.tool.cli/+envspec-filename+)))
  )
