
(ns dobby.tool.cli
  (:import  [java.nio.file Files]
            [java.util Arrays])
  (:require [clojure.edn                 :as edn]
            [clojure.java.io             :as io]
            [clojure.string              :as str]
            [clojure.tools.logging       :as log]
            [clojure.walk                :as walk]
            [cognitect.aws.client.api    :as aws]
            [com.stuartsierra.dependency :as dep]
            [familiar.core               :as f :refer [fmtstr]]
            [dobby.tool.core             :as core]
            [dobby.tool.render           :as render]
            [dobby.tool.render.clj-deps  :as deps-renderer])
  (:gen-class))

(def ^:private +envspec-filename+ ".dobby/env.edn")
(def ^:private +depspec-filename+ ".dobby/dep.edn")
(def ^:private +rendered-depspec-filename+ "Dependencies")

(defn ^:private resolve-depspec-file
  [dir]
  (log/debug "Resolve Dobby dep spec for project %s" dir)
  (.getAbsoluteFile (io/file dir +depspec-filename+)))

(defn ^:private resolve-envspec-file
  "Locate the environment spec for the Dobby workspace relative to `dir`, where `dir` is a (repository) directory
  containing a Dobby dependency template."
  [dir candidates]
  (let [dob-env-file-candidate (io/file dir +envspec-filename+)]
    (log/debug "Resolve Dobby env spec: %s" dob-env-file-candidate)
    (cond (nil? dir)            ; We've run out of parent directories to inspect.
          (throw (ex-info "Dobby environment file not found" {:candidates candidates}))
          (.exists dob-env-file-candidate)
          (.getAbsoluteFile dob-env-file-candidate)
          :else
          (recur (.getParentFile dir) (conj candidates dob-env-file-candidate)))))

(defn ^:private make-depfile-name
  [profile-name]
  (if (= ::core/root profile-name)
    +rendered-depspec-filename+
    (fmtstr "~a.~a" +rendered-depspec-filename+ (name profile-name))))

(defn ^:private compile-project-specs
  [project-dir target-tool]
  (let [depspec-file (resolve-depspec-file project-dir)
        envspec-file (resolve-envspec-file (.getParentFile (.getCanonicalFile depspec-file)) [])]
    ;; FIX: Validate input against a schema/spec so that later processing can be simlified by assuming correctness.
    (doseq [[name pkgspec] (core/reify-deps (io/reader depspec-file) (io/reader envspec-file))]
      (render/render (:dependencies pkgspec)
                     (io/file project-dir (make-depfile-name name))
                     target-tool))))

(defn -main
  "I don't do a whole lot ... yet."
  [& args]
  (cond (= "env-init" (first args))
        (let [account-id   (java.util.UUID. 0 0)
              catalogue-id (java.util.UUID. 0 0)
              pkgdb-bucket "dobby-service-packagedbs3bucket90a06ad6-1fqt68n0ndiju"]
          (letfn [(pair-deps [[node deps]]
                    (loop [deps  deps
                           pairs []]
                      (if (seq deps)
                        (recur (rest deps) (conj pairs [node (first deps)]))
                        pairs)))
                  (build-graph-from-dep-pairs
                    [pairs]
                    (reduce (fn [graph [node dep]]
                              (dep/depend graph node dep))
                            (dep/graph)
                            pairs))
                  (read-catalogue []
                    (->> (aws/invoke (aws/client {:api :s3})
                                     {:op      :GetObject
                                      :request {:Bucket pkgdb-bucket
                                                :Key    (str/join "/" [account-id "catalogue" catalogue-id])}})
                         :Body
                         slurp
                         (edn/read-string {:readers {'com.stuartsierra.dependency.MapDependencyGraph identity}})
                         :dependencies
                         (mapcat pair-deps)
                         build-graph-from-dep-pairs))]
            (spit (second args)
                  {:dobby/environment {:dobby.catalogue/id      catalogue-id
                                       :dobby.catlogue/packages (dep/nodes (read-catalogue account-id catalogue-id))}
                   :dobby/version     {:major "0" :minor "1" :build "0"}})))
        (= "gen-deps" (first args))
        (let [project-dir (.getParent (io/as-file (.getAbsolutePath (io/as-file "."))))]
          (compile-project-specs project-dir ::render/edn)

          (let [files (filter (comp (partial re-matches #"Dependencies.*") #(.getName %))
                              (Arrays/asList (.listFiles (io/as-file project-dir))))]
            (deps-renderer/render (io/as-file project-dir)
                                  (reduce (fn [m f]
                                            (let [profile-name (or (-> f .getName (str/split #"\." 2) second keyword)
                                                                   ::core/root)]
                                              (assoc m profile-name (edn/read-string (slurp f)))))
                                          {}
                                          files))))))
                              ; Note that we're intentionally going through the intermediate files, rather than doing
                              ; everying in memory; eating our own dog food, so to speak.
