;
;Copyright 2013 ºDeme
;
; This file is part of 'dmcljs'.
;
; 'dmcljs' is free software: you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation.
;
; 'dmcljs' is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of 
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with 'dmcljs'. If not, see <http://www.gnu.org/licenses/>.
;
(ns dmcljs.json
  (:use [clojure.string :only [join split replace trim]])
  (:require [cljs.reader :as reader]))


;
; toJson --------------------------------------------------
;

(def ^:private Boolean (type true))
(def ^:private Number (type 1))
(def ^:private String (type ""))
(def ^:private Array (type (array 1 2)))
(def ^:private Seq (type (seq [1 2])))

(defmulti to-json
  "object -> String

  Converts a Clojurescript object to a JSON one.
  
  The clojure object can be: Boolean, Number, String, array, sequence, 
  EmptyList, List, PersistentVector, PersistentHashSet, ObjMap and
  PersistentArrayMap.

  Keys of maps must be keywords 
  "
  (fn [o]
    (let [tp (type o)]
      (cond 
        (= nil o) :null
        (isa? tp Boolean) :boolean
        (isa? tp Number) :number
        (isa? tp String) :string
        (isa? tp Seq) :array
        (isa? tp Array) :array
        (isa? tp cljs.core.EmptyList) :array
        (isa? tp cljs.core.List) :array
        (isa? tp cljs.core.PersistentVector) :array
        (isa? tp cljs.core.PersistentHashSet) :array
        (isa? tp cljs.core.ObjMap) :object
        (isa? tp cljs.core.PersistentArrayMap) :object
        :else :unknow))))

(defmethod to-json :null
  [o]
  "null")

(defmethod to-json :boolean
  [o]
  (if o "true" "false"))

(defmethod to-json :number
  [o]
  (str o))

(defmethod to-json :string
  [o]
  (str "\""
       (replace (replace (replace (replace (replace o
                                                 #"\\" "\\\\")
                                       #"\r" "\\r")
                             #"\n" "\\n")
                   #"\"" "\\\"")
         #"\t" "\\t")
       "\""))

(defmethod to-json :array
  [o]
  (str "[" (clojure.string/join "," (map #(to-json %) (seq o))) "]"))

(defmethod to-json :object
  [o]
  (str "{"
       (clojure.string/join "," (map (fn [[k v]]
                        (when (not (keyword? k))
                          (throw (js/Error.
                                "Map key must be a keyword")))
                        (str (to-json (name k)) ":" (to-json v)))
                      (seq o)))
       "}"))

(defmethod to-json :unknow
  [o]
  (js/console.log o)
  (js/console.log (type o))
  (throw (js/Error.
           (str "Type object is not valid.\nOnly they are valid:\n"
                   "Bolean, Number, String, list, vector, map and nil"))))
               
;
; fromJson ------------------------------------------------
;

(defn- error
  [rst msg]
  (throw (js/Error. 
           (str (clojure.string/join rst) \u0001 \u0002 \u0001 msg))))

(defn- skipBlanks
  [s]
  (drop-while #(<= (.charCodeAt % 0) 32 ) s))

(declare value)

(defn- readNull
  [s]
  (if (= (clojure.string/join (take 4 s)) "null")
    [nil (skipBlanks (drop 4 s))]
    (error s "Error reading 'null' value")))

(defn- readTrue
  [s]
  (if (= (clojure.string/join (take 4 s)) "true")
    [true (skipBlanks (drop 4 s))]
    (error s "Error reading 'true' value")))

(defn- readFalse
  [s]
  (if (= (clojure.string/join (take 5 s)) "false")
    [false (skipBlanks (drop 5 s))]
    (error s "Error reading 'false' value")))

(defn- readString
  [s]
  (loop [v ""
        rst (rest s)]
    (when (= rst '())
      (error rst "Unclosed quotes"))
    (let [ch (first rst)]
      (case ch
        \" [v (skipBlanks (rest rst))]
        \\ (let [rst (rest rst)] 
             (when (= rst '())
               (error rst "Bad escape character"))
             (let [ch (first rst)]
               (case ch
                 (\" \/ \\) (recur (str v ch) (rest rst))
                 \t (recur (str v \tab) (rest rst))
                 \n (recur (str v \newline) (rest rst))
                 \r (recur (str v "\r") (rest rst))
                 \f (recur (str v "\f") (rest rst))
                 \b (recur (str v "\b") (rest rst))
                 \u (let [uch (.fromCharCode js/String
                                (str "0x"
                                  (clojure.string/join (take 4 (rest rst)))))]
                      (when (or (= uch nil)
                                (and (= (.charCodeAt uch. 0) 0)
                                     (not (= (take 4 (rest rst)) 
                                             (seq "0000")))))
                        (error rst "Bad unicode escape sequence"))
                      (recur (str v uch) (drop 5 rst)))
                 (error rst "Bad escape character"))))
        (recur (str v ch) (rest rst))))))

(defn- readNumber
  [s]
  (let [is-digit? 
        (fn [c] (boolean (and (>= c "0") (<= c "9"))))]
    (-> (if (= (first s) \-) ["-" (rest s)] ["" s])
      ((fn [[v rst]]
        (when (= rst '())
          (error rst "Unexpected end of statement reading number"))
        (cond          
          (= (first rst) \0) [(str v \0) (rest rst)]
          (is-digit? (first rst))
            (loop [v2 v
                   rst2 rst]
              (if (or (= rst2 '()) (not (is-digit? (first rst2))))
                [v2 rst2]
                (recur (str v2 (first rst2)) (rest rst2))))
          :else 
            (error rst "Bad intial character"))))
      ((fn [[v rst]]
         (if (or (= rst '()) (not (= (first rst) \.)))
           [v rst]
           (let [v (str v \.)
                 rst (rest rst)]
             (when (= rst '())
               (error rst "Unexpected end of statement reading number"))
             (loop [v v rst rst]
               (if (or (= rst '()) (not (is-digit? (first rst))))
                 [v rst]
                 (recur (str v (first rst)) (rest rst))))))))
      ((fn [[v rst]]
         (if (or (= rst '()) 
                 (and (not (= (first rst) \E))
                      (not (= (first rst) \e))))
           [v rst]
           (let [v (str v \E)
                 rst (rest rst)]
             (when (= rst '())
               (error rst "Unexpected end of statement reading number"))
             (-> (if (or (= (first rst) \-) (= (first rst) \+))
                   [(str v (first rst)) (rest rst)]
                   [v rst])
               ((fn [[v rst]]
                  (when (= rst '())
                    (error
                      rst "Unexpected end of statement reading number"))
                  (loop [v v rst rst]
                    (if (or (= rst '()) (not (is-digit? (first rst))))
                      [v rst]
                      (recur (str v (first rst)) (rest rst)))))))))))                   
      ((fn [[v rst]]
        [(reader/read-string v) (skipBlanks rst)])))))

(defn- readArray
  [s]
  (let [s (skipBlanks (rest s))]
    (cond
      (= s '())
        (error s "Unexpected end of statement reading array")
      (= (first s) \])
        [[] (skipBlanks (rest s))]
      :else
        (loop [v []
               rst s]
          (when (= rst '())
            (error rst "Unexpected end of statement reading array"))
          (let [[e rst] (value rst)]
            (cond
              (= rst '())
                (error rst "Unexpected end of statement reading array")
              (= (first rst) \])
                [(conj v e) (skipBlanks (rest rst))]
              (= (first rst) \,)
                (recur (conj v e) (skipBlanks (rest rst)))
              :else
                (error rst "Missing separator ',' in array")))))))

(defn- readObject 
  [s]
  (let [s (skipBlanks (rest s))]
    (cond
      (= s '())
        (error s "Unexpected end of statement reading object")
      (= (first s) \})
        [{} (skipBlanks (rest s))]
      :else
        (loop [v {}
               rst s]
          (when (= rst '())
            (error rst "Unexpected end of statement reading object"))
          (when (not (= (first rst) \"))
            (error rst "Object key is not string"))
          (let [[[k vl] rst]
                (-> (readString rst)
                  ((fn [[k rst]]
                     (when (= rst '())
                       (error
                         rst "Unexpected end of statement reading object"))
                     (when (not (= (first rst) \:))
                       (error
                         rst "Missing character ':'"))
                     [k (skipBlanks (rest rst))]))
                  ((fn [[k rst]]
                     (let [[v rst] (value rst)]
                       [[k, v] rst]))))]
            (cond
              (= rst '())
                (error rst "Unexpected end of statement reading object")
              (= (first rst) \})
                [(assoc v (keyword k) vl) (skipBlanks (rest rst))]
              (= (first rst) \,)
                (recur (assoc v (keyword k) vl) (skipBlanks (rest rst)))
              :else
                (error rst "Missing separator ',' in object")))))))
            
(defn- value
  [s]
  (when (= s '())
    (error s "Unexpected end of statement"))
  (case (first s)
    \n (readNull s)
    \t (readTrue s)
    \f (readFalse s)
    \" (readString s)
    \[ (readArray s)
    \{ (readObject s)
    (readNumber s)))

(defn from-json 
  "String -> object

  Converts a JSON object to a Clojurescript one.

  The resultant object can be: nil, true, false, Number, String, vector or
  map.

  Keys of map are keywords. 
  "
  [s]
  (try 
    (let [ss (skipBlanks (seq s))]
      (when (= ss '())
        (error ss "Object JSON can't be empty"))
      (let [[v rst] (value ss)]
        (if (= rst '()) 
          v
          (error rst "Extra characters at end of statement"))))    
    (catch js/Error ex
      (if (> (.indexOf (str ex) "\u0001\u0002\u0001" -1)) 
        (let [[rst msg] (split (str ex) #"\u0001\u0002\u0001" 2)]
          (throw (js/Error.
                   (str "[Char " (+ (- (.-length s) (.-length rst)) 7) "] "
                        msg "\n'" s "'"))))
        (throw ex)))))

;
; ajaxSend and receive ------------------------------------
;

(defn json-encode
  "object -> String
  
  Returns (js/encodeURIComponent (to-json s)).
  "
  [s]
  (js/encodeURIComponent (to-json s)))

(defn json-decode
  "String -> object
  
  Returns (from-json (js/decodeURIComponent s))
  "
  [s]
  (from-json (js/decodeURIComponent s)))
  