(ns dmarced.dns          
  (:import [org.apache.commons.lang3.reflect MethodUtils]
           [org.xbill.DNS Lookup Type Name ReverseMap
            Record TXTRecord ARecord AAAARecord PTRRecord
            MXRecord]
           [java.net InetAddress Inet4Address Inet6Address]))



(def sym->type {:txt Type/TXT
                :ptr Type/PTR
                :a Type/A
                :aaaa Type/AAAA
                :mx Type/MX})

(def result->sym {Lookup/HOST_NOT_FOUND :host-not-found
                  Lookup/SUCCESSFUL :success
                  Lookup/TRY_AGAIN :retry
                  Lookup/TYPE_NOT_FOUND :type-not-found
                  Lookup/UNRECOVERABLE :fatal})

(defn lookup-for [type name]
  (let [name (if (= type :ptr) (ReverseMap/fromAddress name) name)]
    (doto (Lookup. name (sym->type type))
      (.run))))

(defmulti transform-record
  class)

(defmethod transform-record TXTRecord [record]
  (MethodUtils/invokeMethod record "getStringsAsByteArrays"))

(defmethod transform-record ARecord [record]
  (.getAddress record))

(defmethod transform-record AAAARecord [record]
  (.getAddress record))

(defmethod transform-record PTRRecord [record]
  (str (.getTarget record)))

(defmethod transform-record MXRecord [record]
  [(.getPriority record) (str (.getTarget record))])

(defn lookup-record [type name]
  (let [lookup (lookup-for type name)
        result (.getResult lookup)
        message (.getErrorString lookup)]
    {:success (= result (Lookup/SUCCESSFUL))
     :result (result->sym result)
     :message message
     :results (for [record (.getAnswers lookup)]
                (transform-record record))}))

(defn- get-domain-addrs [type domain]
  [domain (:results (lookup-record type domain))])

(defn- includes-ip-pred [ip]
  (fn [[_ ips]]
    (let [ips (set ips)]
      (ips ip))))

(defn validated-domains
  "Return the validated domain based on RFC7208 section 5.5"
  [ip]
  (let [type (if (instance? Inet4Address ip) :a :aaaa)
        result (lookup-record :ptr ip)
        results (:results result)]
    {:result (:result result)
     :success (:success result)
     :results (->> results
                   (map (partial get-domain-addrs type))
                   (filter (includes-ip-pred ip))
                   (map first))}))
