(ns portal.debug)

(def ^:dynamic *output* nil)

(defn- dispatch [form]
  (if-not (list? form) form (first form)))

(defn now []
  #?(:clj (java.util.Date.) :cljs (js/Date.)))

(defn- capture [label form]
  (let [{:keys [line column]} (meta label)]
    `(let [[flow# result#]
           (try
             [:result ~form]
             (catch #?(:clj Exception :cljs :default) ex#
               [:throw ex#]))]
       (swap! *output* conj
              {:form     (quote ~label)
               :flow     flow#
               :result   result#
               :ns       (quote ~(symbol (str *ns*)))
               :line     (quote ~line)
               :column   (quote ~column)
               :time     (now)})
       (if (= flow# :result) result# (throw result#)))))

(defmulti -trace #'dispatch)

(defmethod -trace :default [form] (capture form form))

(defmethod -trace 'let [form]
  (let [[_ bindings & body] form]
    (capture
     form
     `(let
       ~(->> (partition 2 bindings)
             (mapcat
              (fn [[binding expr]]
                [binding (capture binding expr)]))
             (into []))
        ~@body))))

(defmethod -trace 'if [form]
  (let [[_ condition a b] form]
    (capture
     form
     `(if ~(capture condition condition) ~a ~b))))

(defmethod -trace 'cond [form]
  (let [[_ & conditions] form]
    (capture
     form
     `(cond
        ~@(mapcat
           (fn [[condition expr]]
             [(capture condition condition) expr])
           (partition 2 conditions))))))

(defmacro trace [form]
  `(binding [*output* (atom [])]
     (try
       (let [result# ~(-trace form)]
         (tap> @*output*)
         result#)
       (catch #?(:clj Exception :cljs :default) ex#
         (tap> @*output*)
         (throw ex#)))))
