(ns numords.core
  "Convert Western Arabic numbers to words, including decimal points."
  {:author ["David Harrigan"]}
  (:require
   [clojure.string :refer [join split]]))

(set! *assert* true)
(set! *warn-on-reflection* true)

(def ^:private units
  {0 "zero"
   1 "one"
   2 "two"
   3 "three"
   4 "four"
   5 "five"
   6 "six"
   7 "seven"
   8 "eight"
   9 "nine"})

(def ^:private first-twenty
  (merge units {10 "ten"
                11 "eleven"
                12 "twelve"
                13 "thirteen"
                14 "fourteen"
                15 "fifteen"
                16 "sixteen"
                17 "seventeen"
                18 "eighteen"
                19 "nineteen"}))

(def ^:private tens
  ["zero"
   "ten"
   "twenty"
   "thirty"
   "forty"
   "fifty"
   "sixty"
   "seventy"
   "eighty"
   "ninety"])

(def ^:private powers
  ["hundred" "thousand" "million"])

(defn ^:private qr
  ([number base] (qr number base nil))
  ([number base prefix]
   (if prefix
     [(prefix (quot number base)) (rem number base)]
     [(quot number base) (rem number base)])))

(defn ^:private qr-tens
  [number]
  (qr number 10 tens))

(defn ^:private qr-hundreds
  [number]
  (qr number 100 units))

(defn ^:private convert
  [number]
  (cond
   (= number 0) "zero"
   (< number 20) (first-twenty number)
   (< number 100) (let [[prefix remainder] (qr-tens number)]
                    (if (zero? remainder) prefix (str prefix " " (convert remainder))))
   (< number 1000) (let [[prefix remainder] (qr-hundreds number)]
                     (str prefix " hundred" (when-not (zero? remainder) (str " and " (convert remainder)))))
   :else (loop [number number
                grouping []
                power 0]
           (if (zero? number)
             (join " " grouping)
             (let [[quotient remainder] (qr number 1000)
                   grouping (if (zero? remainder)
                              grouping
                              (let [cardinal (convert remainder)]
                                (cons (cond
                                       (zero? power) (if (> 100 remainder) (str "and " cardinal) cardinal)
                                       :else (str cardinal " " (powers power))) grouping)))]
               (recur quotient grouping (inc power)))))))

(defn number->words
  "Convert Western Arabic numbers into words by taking an `integer` (in the mathematical scene)
   from -n, 0, n and converting it into it's respective British English words. It can work with
   decimal points too.

   Some examples:

   ```clojure
   (number->words -1) ;; \"minus one\"
   (number->words 0) ;; \"zero\"
   (number->words 999) ;; \"nine hundred and ninety nine\"
   (number->words 999.1) ;; \"nine hundred and ninety nine point one\"
   (number->words 1000) ;; \"one thousand\"
   (number->words 2500) ;; \"two thousand five hundred\"
   (number->words 3333.345) ;; \"three thousand three hundred and thirty three point three four five\"
   ```
   "
  [number]
  {:pre [(and (number? number)
              (not (> number 999999999)))]}
  (let [[integer fraction] (split (str number) #"\.")
        integer-words (convert (Math/abs ^long (Long/parseLong integer)))
        fractional-words (map #(convert (Character/getNumericValue ^char %)) fraction)
        combined-words (str integer-words (when (seq fractional-words)
                                            (str " point " (join " " fractional-words))))]
    (if (neg? number)
      (str "minus " combined-words)
      combined-words)))

(comment

 (require '[numords.core :as c])

 (rem 999 100)
 (quot 1999 1000)
 (rem 1999 1000)
 (rem 1005 1000)

 (c/qr 99 10)
 (c/qr 100 10)
 (c/qr 999 100)

 (c/number->words -1000105) ;; "minus one million one hundred and five"
 (c/number->words -99.001) ;; "minus ninety nine point zero zero one"
 (c/number->words -45)
 (c/number->words -1) ;; "minus one"
 (c/number->words 0) ;; "zero"
 (c/number->words 0.1) ;; "zero point one"
 (c/number->words 10) ;; "ten"
 (c/number->words 15) ;; "fifteen"
 (c/number->words 20)
 (c/number->words 21) ;; "twenty one"
 (c/number->words 40) ;; "forty"
 (c/number->words 45) ;; "forty five"
 (c/number->words 53) ;; "fifty three"
 (c/number->words 90)
 (c/number->words 99) ;; "ninety nine"
 (c/number->words 99.1) ;; "ninety nine point one"
 (c/number->words 100) ;; "one hundred"
 (c/number->words 101) ;; "one hundred and one"
 (c/number->words 205)
 (c/number->words 333) ;; "three hundred and thirty three"
 (c/number->words 500)
 (c/number->words 999) ;; "nine hundred and ninety nine"
 (c/number->words 1000) ;; "one thousand"
 (c/number->words 1005)
 (c/number->words 1040) ;; "one thousand and forty"
 (c/number->words 1045)
 (c/number->words 1999) ;; "one thousand nine hundred and ninety nine"
 (c/number->words 2500) ;; "two thousand five hundred"
 (c/number->words 3333.345) ;; "three thousand three hundred and thirty three point three four five"
 (c/number->words 1000105) ;; "one million one hundred and five"
 (c/number->words 10000105)
 (c/number->words 999999999)

 ,)
