;; Copyright (c) Brenton Ashworth. All rights reserved.
;; The use and distribution terms for this software are covered by the
;; Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; which can be found in the file COPYING at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by
;; the terms of this license.
;; You must not remove this notice, or any other, from this software.

(ns deview.test-reporting
  "Capture test results and post them to the server for this project."
  (:use (clj-stacktrace core)
        (deview project))
  (:require [clojure.test :as ct])
  (:import (java.io BufferedWriter
                    OutputStreamWriter)
           (java.net Socket)))

(defmulti process-report :type)

(defmethod process-report :default [m] m)

(defmethod process-report :begin-test-ns [m]
  (assoc m :ns (str (:ns m))))

(defmethod process-report :end-test-ns [m]
  (assoc m :ns (str (:ns m))))

(defmethod process-report :begin-test-var [m]
  (assoc m :var (str (:var m))))

(defmethod process-report :end-test-var [m]
  (assoc m :var (str (:var m))))

(defmethod process-report :pass [m]
  {:type :pass})

(defn assoc-as-str [m k]
  (assoc m k (str (k m))))

(defn assoc-file-position [m]
  (let [[file line] (ct/file-position 4)]
    (-> m
        (assoc :file file)
        (assoc :line line))))

(defmethod process-report :fail [m]
  (let [[file line] (ct/file-position 4)]
    (-> (if (seq ct/*testing-contexts*)
         (assoc m :context-message (ct/testing-contexts-str))
         m)
       (assoc-as-str :actual)
       (assoc-as-str :expected)
       (assoc :file file)
       (assoc :line line))))

(defmethod process-report :error [m]
  (let [actual (:actual m)
        [file line] (ct/file-position 4)]
    (-> (if (instance? Throwable actual)
          (assoc m :actual (parse-exception actual))
          (assoc-as-str m :actual))
        (assoc-as-str :expected)
        (assoc :file file)
        (assoc :line line))))

(def report-order (atom 0))

(defn send-report
  "Send test report to the server."
  [report namespace port]
  (with-open [socket (Socket. "localhost" port)
              writer (BufferedWriter.
                      (OutputStreamWriter.
                       (.getOutputStream socket)))]
    (do (.write writer (str {:action :post-test-result
                             :namespace namespace
                             :value (assoc report :order
                                           (swap! report-order inc))}))
        (.flush writer))))

(defn status-recorder
  "Create a function to replace the default clojure.test/report function. It
   will capture test results process them and send them to the server
   for this project."
  [namespace]
  (let [port (:deview-server (read-project))]
    (fn
      [m]
      (-> (process-report m)
          (send-report namespace port)))))

(defn safe-require
  "Try to require a namespace. Catch any exception and send the report to
   the server."
  [namespace]
  (try
    (require namespace)
    true
    (catch Exception e
      (do
        (send-report {:exception (parse-exception e)}
                     namespace
                     (:deview-server (read-project)))
        false))))

(defn run-tests
  "Run tests for a namespace."
  [namespace]
  (binding [ct/report (status-recorder namespace)]
    (ct/run-tests namespace)))
