;; Copyright (c) Brenton Ashworth. All rights reserved.
;; The use and distribution terms for this software are covered by the
;; Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; which can be found in the file COPYING at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by
;; the terms of this license.
;; You must not remove this notice, or any other, from this software.

(ns deview.test
  "Run tests in a Clojure project. Outsource the actual running to
   Leiningen."
  (:refer-clojure :exclude [test])
  (:use [clojure.java.io :only [file]]
        [leiningen.compile :only [eval-in-project]]
        (deview project))
  (:import (java.io File)))


;; These are the new refs that you will use to manage test state.
;;
;; test-queue is a list of namespaces that need to be tested. Create a
;; test loop that will run these tests one at a time.
;; test-results is a map of test results:
;; {"ns-name" {1 {:timestamp 1
;;                :status <fn>
;;                :result <future>}
;;            {2 {...}}}
(def next-test-id (ref 1))
(def test-queue (ref [(promise)]))
(def test-results (ref {}))

(defn- add-ns-to-test-results
  ([m ns id ts recorder]
     (add-ns-to-test-results m ns id ts recorder :pending))
  ([m ns id ts recorder result]
     (-> m
         (assoc-in
          [ns id]
          {:id id
           :status recorder
           :result result
           :timestamp ts}))))

(defn- add-ns-to-test-queue
  "The last element in the queue is always a promise to deliver the next
   namespace. This function will fulfill that promise as well as add a new
   promise to queue."
  [ns id ts]
  (dosync
   (let [next-promise (last @test-queue)]
     (alter test-queue conj (promise))
     (deliver next-promise {:ns ns :id id :ts ts}))))

(defn- modified-files?
  "Have any files in this project been modified after this test was started?"
  [results]
  (let [{files :message} (modified-files {:timestamp (:timestamp results)})]
    (> (count files) 0)))

(defn- test-result-exists?
  "Are there any test results for this namespace in the test-results map?
   Returns true or false."
  [ns]
  (get @test-results ns))

(defn- get-id-from-queue
  "If this namespace exists in the queue, return its id, otherwise return
   nill."
  [namespace]
  (if-let [items (butlast @test-queue)]
    (->> items
         (map deref)
         (filter #(= (:ns %) namespace))
         last
         :id)))

(defn- start-new-test
  "First check to see if this namespace is already in the queue. If it is
   then return the id of this test. If it is not in the queue then add it.
   Always returns a test id."
  [namespace ts]
  (if-let [id (get-id-from-queue namespace)]
    id
    (let [id (alter next-test-id inc)]
      (add-ns-to-test-queue namespace id ts)
      id)))

(defn- keep-test-results? [most-recent-ts [_ v]]
  (or (= (:result v) :pending)
      (> (:timestamp v) most-recent-ts)))

(defn- delete-old-result
  "Delete any test results that are finished and have older timestamps than
   the most recent file modification time."
  [namespace result]
  (let [results (get @test-results namespace)
        ts (:timestamp result)
        modified-results
        (apply hash-map
               (apply concat
                      (filter (partial keep-test-results? ts)
                              results)))]
    (if (empty? modified-results)
      (alter test-results dissoc namespace)
      (alter test-results
             assoc
             namespace
             modified-results))))

(defn- package-result [result]
  (-> (dissoc result :status)
      (assoc :status ((:status result)))))

(defn get-test-result
  "Get the results for this namespace."
  ([namespace]
     (get-test-result namespace nil))
  ([namespace id]
     (if-let [results (get @test-results namespace)]
       (let [id (or id (apply max (keys results)))
             result (get results id)]
         (package-result result)))))

(defn- existing-test-results
  "Return existing test results or the id of tests that are running."
  [ns result ts]
  (cond (modified-files? result)
        (do (delete-old-result ns result)
            (start-new-test ns ts))
        (= (:result result) :pending)
        (:id result)
        :else result))

(defn start-test!
  "Queue this namespace to be tested. This will either return test results for
   a completed test or the id of a test that is running or will be run."
  [ns]
  (dosync
   (let [ts (System/currentTimeMillis)]
     (if-let [result (get-test-result ns)]
       (existing-test-results ns result ts)
       (start-new-test ns ts)))))

(defn status-recorder
  "Create a function that can accept a new status record or report on
   the current status."
  []
  (let [status (atom [])]
    (fn
      ([] @status)
      ([m]
         (swap! status conj m)))))

(defn record-test-result
  "Record this test result in the status for this namespace."
  [test-result-record namespace]
  (let [results (get @test-results (name namespace))
        id (apply max (keys results))
        result (get results id)]
    ((:status result) test-result-record)))

(defn form-for-testing-namespace
  [namespace test-package]
  `(do
     (require '~test-package)
     (let [resolver# (fn [fname#]
                       (ns-resolve
                        (find-ns '~test-package) fname#))]
       (if ((resolver# ~''safe-require) '~namespace)
         (let [summary# ((resolver# ~''run-tests) '~namespace) ]
           (when-not (= "1.5" (System/getProperty "java.specification.version"))
             (shutdown-agents)))))
     (System/exit 0)))

(defn run-test-with-lein
  "Use Leiningen to run tests in the project with the project's classpath."
  [project namespace]
  (do
    (eval-in-project project
                     (form-for-testing-namespace (symbol namespace)
                                                 'deview.test-reporting))
    0))

(defn test-namespace
  "Test one namespace. Blocks until tests are finished."
  [project ns id ts]
  (let [recorder (status-recorder)]
    (dosync (alter test-queue #(vec (rest %)))
            (alter test-results
                   add-ns-to-test-results
                   ns
                   id
                   ts
                   recorder))
    (do (run-test-with-lein project ns)
        (Thread/sleep 500)
        (let [status (recorder)
              result (if (= (count status) 1)
                       (first status)
                       (first (filter #(= (:type %) :summary) (recorder))))]
          (dosync (alter test-results
                         assoc-in
                         [ns id :result]
                         result))))))

(defn sequential-test-runner
  "Run each test in the queue, one at a time. As long as you use
   add-ns-to-test-queue to add items the queue, it will always contain
   at least one promise."
  [project]
  (if (seq @test-queue)
    (let [{:keys [id ts ns]} @(first @test-queue)]
      (test-namespace project ns id ts)
      (recur project))))


