;; Copyright (c) Brenton Ashworth. All rights reserved.
;; The use and distribution terms for this software are covered by the
;; Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; which can be found in the file COPYING at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by
;; the terms of this license.
;; You must not remove this notice, or any other, from this software.

(ns deview.server
  (:require (deview [test :as test]
                    [project :as project]
                    [metrics :as metrics]))
  (:import (java.io BufferedReader
                    BufferedWriter
                    InputStreamReader
                    OutputStreamWriter)
           (java.net Socket 
                     ServerSocket)))

(def version "1.2.0")

(def server (atom nil))

(defmulti execute-command :action)

(defmethod execute-command :default [_]
           {:message "unknown command"})

(defmethod execute-command :test [{:keys [namespace]}]
           (let [status (test/start-test! namespace)]
             (if (number? status)
               {:status "started" :id status}
               status)))

(defmethod execute-command :post-test-result [{:keys [value namespace]}]
           (test/record-test-result value namespace))

(defmethod execute-command :test-results [{:keys [namespace id]}]
           (if-let [test-result (test/get-test-result namespace id)]
             test-result
             (execute-command {:action :test :namespace namespace})))

(defmethod execute-command :info [{:keys [type] :as m}]
           (cond (= type :test-namespaces) (project/test-namespaces)
                 (= type :project) (project/read-project)
                 (= type :metrics) (metrics/run-metrics
                                    (-> metrics/default-metric
                                        metrics/loc-metric))
                 (= type :version) {:version version}
                 (= type :modified) (project/modified-files m)
                 :else {:message "unknown info"}))

(defmethod execute-command :ping [_]
           {:message "pong"})

(defn- handle-connection [conn]
  (with-open [reader (BufferedReader.
                      (InputStreamReader. (.getInputStream conn)))
              writer (BufferedWriter.
                      (OutputStreamWriter. (.getOutputStream conn)))]
    (loop [request (.readLine reader)]
     (if request
       (if-let [response (execute-command (read-string request))]
         (let [response (cond (string? response) response
                              (map? response) (str response)
                              :else (str (into [] response)))]
           (do (.write writer response)
               (.write writer "\r\n")
               (.flush writer)
               (recur (.readLine reader)))))
       (do (.close conn))))))

(defn- server-loop [server-socket]
  (let [conn (.accept server-socket)]
    (do (.setKeepAlive conn true)
        (future (handle-connection conn))
        (recur server-socket))))

(defn start [project port]
  (do (println "Starting Server on Port:" port)
      (println project)
      (let [ss (ServerSocket. port)]
        (future (server-loop ss))
        (swap! server
               (fn [a b] b)
               {:server-socket ss
                :test-runner (future (test/sequential-test-runner project))
                :port port
                :project project}))))

(defn stop [server]
  (.close (:server-socket server)))

(defmethod execute-command :shutdown [_]
           (do (.close (:server-socket @server))
               (System/exit 0)))

