;; Copyright (c) Brenton Ashworth. All rights reserved.
;; The use and distribution terms for this software are covered by the
;; Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; which can be found in the file COPYING at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by
;; the terms of this license.
;; You must not remove this notice, or any other, from this software.

(ns deview.project
  (:use [clojure.contrib.find-namespaces
         :only (find-clojure-sources-in-dir
                read-file-ns-decl)])
  (:import (java.io File)))

(defn read-project []
  (let [[_ name v & m] (read-string (slurp "project.clj"))]
    (merge {:name (str name)
            :version (str v)}
           (apply hash-map m))))

(defn get-test-path
  ([] (get-test-path (read-project)))
  ([p] (get p :test-path "test")))

(defn get-source-path
  ([] (get-source-path (read-project)))
  ([p] (get p :source-path "src")))

;; I lifted the four functions below from Stuart Sierra's lazytest

(defn find-sources
  [dirs]
  {:pre [(coll? dirs)
         (every? (fn [d] (instance? java.io.File d)) dirs)]}
  (mapcat find-clojure-sources-in-dir dirs))

(defn namespace-for-file [f]
  (second (read-file-ns-decl f)))

(defn newer-sources [dirs timestamp]
  (filter #(> (.lastModified %) timestamp) (find-sources dirs)))

(defn newer-namespaces [dirs timestamp]
  (map str
       (remove nil? (map namespace-for-file (newer-sources dirs timestamp)))))

(defn clj-dirs []
  [(File. (get-source-path)) (File. (get-test-path))])

(defn modified-files [m]
  (let [ts (get m :timestamp 0)
        new-ts (System/currentTimeMillis)
        new-ns (newer-namespaces (clj-dirs) ts)]
    {:message new-ns :timestamp new-ts}))

;;
;; Finding Tests
;;

(defn- possible-test-file? [file-name]
  (and (if (re-matches #"\w+[.]{1}clj" file-name) true false)
       (not (.startsWith file-name "_"))))

(defn- get-namespace-name [forms]
  (str (second (first (filter #(= (first %) 'ns) forms)))))

(defn test-namespaces
  "Get a list of all test namespaces (as strings) for the current
   project."
  []
  (->> (file-seq (File. (get-test-path)))
       (filter #(possible-test-file? (.getName %)))
       (map #(read-string (str "[" (slurp (.getAbsolutePath %)) "]")))
       (reduce (fn [a b]
                 (if (some #(= (first %) 'deftest) b)
                   (conj a (get-namespace-name b))
                   a))
               [])))

