;; Copyright (c) Brenton Ashworth. All rights reserved.
;; The use and distribution terms for this software are covered by the
;; Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; which can be found in the file COPYING at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by
;; the terms of this license.
;; You must not remove this notice, or any other, from this software.

(ns deview.test
  (:use (clojure test)
        (clj-stacktrace core)
        (deview project))
  (:import (java.io File)))

(defn possible-test-file? [file-name]
  (and (if (re-matches #"\w+[.]{1}clj" file-name) true false)
       (not (.startsWith file-name "_"))))

(defn get-namespace-name [forms]
  (str (second (first (filter #(= (first %) 'ns) forms)))))

(defn test-namespaces
  "Get a list of all test namespaces (as strings) for the current
   project."
  []
  (->> (file-seq (File. (get-test-path)))
       (filter #(possible-test-file? (.getName %)))
       (map #(read-string (str "[" (slurp (.getAbsolutePath %)) "]")))
       (reduce (fn [a b]
                 (if (some #(= (first %) 'deftest) b)
                   (conj a (get-namespace-name b))
                   a))
               [])))

(def test-results (ref {}))

(defmulti process-report :type)

(defmethod process-report :default [m] m)

(defmethod process-report :begin-test-ns [m]
  (assoc m :ns (str (:ns m))))

(defmethod process-report :end-test-ns [m]
  (assoc m :ns (str (:ns m))))

(defmethod process-report :begin-test-var [m]
  (assoc m :var (str (:var m))))

(defmethod process-report :end-test-var [m]
  (assoc m :var (str (:var m))))

(defmethod process-report :pass [m]
  {:type :pass})

(defn assoc-as-str [m k]
  (assoc m k (str (k m))))

(defn assoc-file-position [m]
  (let [[file line] (file-position 4)]
    (-> m
        (assoc :file file)
        (assoc :line line))))

(defmethod process-report :fail [m]
  (let [[file line] (file-position 4)]
    (-> (if (seq *testing-contexts*)
         (assoc m :context-message (testing-contexts-str))
         m)
       (assoc-as-str :actual)
       (assoc-as-str :expected)
       (assoc :file file)
       (assoc :line line))))

(defmethod process-report :error [m]
  (let [actual (:actual m)
        [file line] (file-position 4)]
    (-> (if (instance? Throwable actual)
          (assoc m :actual (parse-exception actual))
          (assoc-as-str m :actual))
        (assoc-as-str :expected)
        (assoc :file file)
        (assoc :line line))))

(defn status-recorder
  "Create a function that can capture and store test reports. When called
   with a report map it will process and store it. When called with no
   argument it will return the current status."
  []
  (let [status (atom [])]
    (fn
      ([] @status)
      ([m]
         (swap! status conj (process-report m))))))

(defn test-results!
  "Get the current test results for this namespace. If the test is finished
   then remove it from the test results."
  [namespace]
  (if-let [m (get @test-results namespace)]
    (let [status ((:status m))
          result (:result m)]
      (if (future-done? result)
        (do (dosync (alter test-results dissoc namespace))
            (-> {}
                (assoc :status status)
                (assoc :result @result)))
        (-> {}
            (assoc :status status)
            (assoc :result :pending))))
    {:message (str "no results for " namespace)}))

(defn test-running? [m namespace]
  (get m namespace))

;; TODO - Read the source file to get the current list of tests and
;; unmap any that are in the namespace but not in the file.
;;
;; (ns-unmap 'deview.test-app 'test-something)

(defn reload-namespace [namespace]
  (try
   (require :reload-all (symbol namespace))
   (catch Exception e {:exception (parse-exception e)})))

(defn test-namespace [namespace recorder]
  {:summary (binding [report recorder]
              (run-tests (symbol namespace)))})

;; I am using a ref instead of an atom because I need to check that a
;; test is running and then create a future within a transaction.

;; TODO - Re-create the problem that you had and post to Clojure
;; group. Could be a bug. Put a future in an atom where the function
;; inside the future throws an exception that is caught. Works if you
;; wait until the future completes but not while it is pending.

(defn add-ns-to-test-results [m ns recorder result]
  (-> m
      (assoc ns {:status recorder
                 :result result})))

(defn start-test [namespace]
  (let [recorder (status-recorder)]
    (dosync
     (if (not (test-running? @test-results namespace))
       (if-let [load-result (reload-namespace namespace)]
         (alter test-results
              add-ns-to-test-results
              namespace
              recorder
              (future load-result))
         (alter test-results
              add-ns-to-test-results
              namespace
              recorder
              (future (test-namespace namespace recorder))))))))

;; Writing the above function in a more consice way, as shown below,
;; does not work.

(comment
  (defn start-test [namespace]
    (let [recorder (status-recorder)]
      (dosync
       (if (not (test-running? @test-results namespace))
         (alter test-results
                add-ns-to-test-results
                namespace
                recorder
                (if-let [load-result (reload-namespace namespace)]
                  (future load-result)
                  (future (test-namespace namespace recorder)))))))))

