;; Copyright (c) Brenton Ashworth. All rights reserved.
;; The use and distribution terms for this software are covered by the
;; Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; which can be found in the file COPYING at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by
;; the terms of this license.
;; You must not remove this notice, or any other, from this software.

(ns deview.metrics
  "Various project metrics."
  (:use (deview project))
  (:import (java.io File
                    BufferedReader
                    InputStreamReader
                    FileInputStream)))

;; If we "use" this function from clojure.contrib.map-utils and a project
;; depends on clojure 1.2 but doesn't use contrib, deview will not
;; work. The project owner would have to add contrib 1.2 as a
;; dependency. Instead of requiring this of project owners, for now, I am
;; including the one function that I need here.

;; TODO: Bring this issue up on the mailing list

(defn deep-merge-with
  "Like merge-with, but merges maps recursively, applying the given fn
only when there's a non-map at a particular level.

(deepmerge + {:a {:b {:c 1 :d {:x 1 :y 2}} :e 3} :f 4}
{:a {:b {:c 2 :d {:z 9} :z 3} :e 100}})
-> {:a {:b {:z 3, :c 3, :d {:z 9, :x 1, :y 2}}, :e 103}, :f 4}"
  [f & maps]
  (apply
    (fn m [& maps]
      (if (every? map? maps)
        (apply merge-with m maps)
        (apply f maps)))
    maps))

(defn non-blank? [line] (if (re-find #"\S" line) true false))

(defn file-ext
  "Get the lower case file extension from the file name. If the file does not
   have an extension then return nil."
  [file-path]
  (let [ext-string (.toLowerCase
                    (.substring file-path (+ 1 (.lastIndexOf file-path "."))))]
    (if (contains? (set ext-string) \/)
      nil
      ext-string)))

(defn- project-root-dir [p]
  (.getParentFile (File. (.getAbsolutePath (File. (get-source-path p))))))

(defn file-filter
  "Filter out non-source files."
  [f]
  (let [{loc-exts :deview-loc-ext} (read-project)
        path (.getAbsolutePath f)
        name (.getName f)]
    (and (not (.isDirectory f))
         (not (.startsWith name "."))
         (not (.startsWith name "#"))
         (let [ext (file-ext name)]
           (if loc-exts
             (contains? loc-exts ext)
             (not (contains? #{"class" "jar" "gif" "jpg" "png"} ext)))))))

(defn process-files
  "Apply the handler function to the text-string of each file in the project."
  [handler p]
  (apply deep-merge-with +
         (filter #(not (nil? %))
                 (map (fn [[file string]]
                        (handler p file string))
                      (map #(vector % (slurp (.getAbsolutePath %)))
                           (filter file-filter
                                   (file-seq (project-root-dir p))))))))

(defn run-metrics
  "Run all metrics in the handler function returning a map of calculated
   metrics."
  [handler]
  (let [p (read-project)]
    (process-files handler p)))

(defn metric-location
  "Projects files are located in various locations. Return the location for
   this file. One of #{source test other}"
  [p file-path]
  (let [source-path (.getAbsolutePath (File. (get-source-path p)))
        test-path (.getAbsolutePath (File. (get-test-path p)))]
    (cond (.startsWith file-path source-path) "source"
          (.startsWith file-path test-path) "test"
          :else "other")))

(defn default-metric [p file string]
  {:file-count 1})

(defn loc-metric [handler]
  (fn [p file string]
    (let [file-path (.getAbsolutePath file)
          ext (file-ext file-path)
          location (metric-location p file-path)]
      (when ext
        (merge (handler p file string)
               {(keyword location)
                {:loc {(keyword ext)
                       (count (filter non-blank?
                                      (seq (.split string "\n"))))}}})))))
