;; Copyright (c) Brenton Ashworth. All rights reserved.
;; The use and distribution terms for this software are covered by the
;; Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; which can be found in the file COPYING at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by
;; the terms of this license.
;; You must not remove this notice, or any other, from this software.

(ns deview.server
  (:require (deview [test :as test]
                    [project :as project]
                    [stats :as stats]))
  (:import (java.io BufferedReader
                    BufferedWriter
                    InputStreamReader
                    OutputStreamWriter)
           (java.net Socket 
                     ServerSocket)))

(defmulti execute-command first)

(defmethod execute-command :default [_]
  {:message "unknown command"})

(defmethod execute-command "test" [[_ namespace]]
  (do (test/start-test namespace)
      {:message "running"}))

(defmethod execute-command "test-results" [[_ namespace]]
  (test/test-results! namespace))

(defmethod execute-command "info" [[_ info]]
  (cond (= info "test-namespaces") (test/test-namespaces)
        (= info "project") (project/read-project)
        (= info "stats") (stats/project-statistics)
        :else {:message "unknown info"}))

(defn- handle-command [c]
  (if (= c "ping")
    {:message "pong"}
    (execute-command (seq (.split c ":")))))

(defn- handle-connection [conn]
  (with-open [reader (BufferedReader.
                      (InputStreamReader. (.getInputStream conn)))
              writer (BufferedWriter.
                      (OutputStreamWriter. (.getOutputStream conn)))]
    (loop [request (.readLine reader)]
     (if request
       (if-let [response (handle-command request)]
         (let [response (cond (string? response) response
                              (map? response) (str response)
                              :else (str (into [] response)))]
           (do (.write writer response)
               (.write writer "\r\n")
               (.flush writer)
               (recur (.readLine reader)))))
       (do (.close conn))))))

(defn- server-loop [server-socket]
  (let [conn (.accept server-socket)]
    (do (.setKeepAlive conn true)
        (future (handle-connection conn))
        (recur server-socket))))

(defn start [port]
  (do (println "Starting Server on Port:" port)
      (let [ss (ServerSocket. port)]
        (future (server-loop ss))
        {:server-socket ss
         :port port})))

(defn stop [server]
  (.close (:server-socket server)))
