(ns cljfmt.core
  #?(:clj (:refer-clojure :exclude [reader-conditional?]))
  (:require #?(:clj [clojure.java.io :as io])
            [clojure.string :as str]
            [rewrite-clj.node :as n]
            [rewrite-clj.parser :as p]
            [rewrite-clj.zip :as z])
  #?(:clj (:import java.util.regex.Pattern)
     :cljs (:require-macros [cljfmt.core :refer [read-resource]])))

#?(:clj (def read-resource* (comp read-string slurp io/resource)))
#?(:clj (defmacro read-resource [path] `'~(read-resource* path)))

(def includes?
  #?(:clj  (fn [^String a ^String b] (.contains a b))
     :cljs str/includes?))

#?(:clj
   (defn- find-all [zloc p?]
     (loop [matches []
            zloc zloc]
       (if-let [zloc (z/find-next zloc z/next* p?)]
         (recur (conj matches zloc)
                (z/next* zloc))
         matches))))

(defn- edit-all [zloc p? f]
  (loop [zloc (if (p? zloc) (f zloc) zloc)]
    (if-let [zloc (z/find-next zloc z/next* p?)]
      (recur (f zloc))
      zloc)))

(defn- transform [form zf & args]
  (z/root (apply zf (z/of-node form) args)))

(defn root? [zloc]
  (nil? (z/up* zloc)))

(defn- top? [zloc]
  (some-> zloc z/up root?))

(defn- root [zloc]
  (if (root? zloc) zloc (recur (z/up zloc))))

(defn- clojure-whitespace? [zloc]
  (z/whitespace? zloc))

(defn- unquote? [zloc]
  (and zloc (= (n/tag (z/node zloc)) :unquote)))

(defn- deref? [zloc]
  (and zloc (= (n/tag (z/node zloc)) :deref)))

(defn- unquote-deref? [zloc]
  (and (deref? zloc)
       (unquote? (z/up* zloc))))

(defn- comment? [zloc]
  (some-> zloc z/node n/comment?))

(defn- surrounding-whitespace? [zloc]
  (and (not (top? zloc))
       (clojure-whitespace? zloc)
       (or (and (nil? (z/left* zloc))
                ;; don't convert ~ @ to ~@
                (not (unquote-deref? (z/right* zloc)))
                ;; ignore space before comments
                (not (comment? (z/right* zloc))))
           (nil? (z/skip z/right* clojure-whitespace? zloc)))))

(defn remove-surrounding-whitespace [form]
  (transform form edit-all surrounding-whitespace? z/remove*))

(defn- element? [zloc]
  (and zloc (not (z/whitespace-or-comment? zloc))))

(defn- reader-macro? [zloc]
  (and zloc (= (n/tag (z/node zloc)) :reader-macro)))

(defn- namespaced-map? [zloc]
  (and zloc (= (n/tag (z/node zloc)) :namespaced-map)))

(defn- missing-whitespace? [zloc]
  (and (element? zloc)
       (not (reader-macro? (z/up* zloc)))
       (not (namespaced-map? (z/up* zloc)))
       (element? (z/right* zloc))))

(defn insert-missing-whitespace [form]
  (transform form edit-all missing-whitespace? z/insert-space-right))

(defn- space? [zloc]
  (= (z/tag zloc) :whitespace))

(defn- line-comment? [zloc]
  (and (comment? zloc) (re-matches #"(?s);;([^;].*)?" (z/string zloc))))

(defn- comma? [zloc]
  (some-> zloc z/node n/comma?))

(defn- line-break? [zloc]
  (or (z/linebreak? zloc) (comment? zloc)))

(defn- skip-whitespace [zloc]
  (z/skip z/next* space? zloc))

(defn- skip-whitespace-and-commas [zloc]
  (z/skip z/next* #(or (space? %) (comma? %)) zloc))

(defn- skip-clojure-whitespace
  ([zloc] (skip-clojure-whitespace zloc z/next*))
  ([zloc f] (z/skip f clojure-whitespace? zloc)))

(defn- count-newlines [zloc]
  (loop [zloc' zloc, newlines 0]
    (if (z/linebreak? zloc')
      (recur (-> zloc' z/right* skip-whitespace-and-commas)
             (-> zloc' z/string count (+ newlines)))
      (if (comment? (skip-clojure-whitespace zloc z/left*))
        (inc newlines)
        newlines))))

(defn- final-transform-element? [zloc]
  (nil? (skip-clojure-whitespace (z/next* zloc))))

(defn- consecutive-blank-line? [zloc]
  (and (> (count-newlines zloc) 2)
       (not (final-transform-element? zloc))))

(defn- remove-clojure-whitespace [zloc]
  (if (clojure-whitespace? zloc)
    (recur (z/remove* zloc))
    zloc))

(defn- replace-consecutive-blank-lines [zloc]
  (let [zloc-elem-before (-> zloc
                             skip-clojure-whitespace
                             z/prev*
                             remove-clojure-whitespace)]
    (-> zloc-elem-before
        z/next*
        (z/insert-left* (n/newlines (if (comment? zloc-elem-before) 1 2))))))

(defn remove-consecutive-blank-lines [form]
  (transform form edit-all consecutive-blank-line? replace-consecutive-blank-lines))

(defn- indentation? [zloc]
  (and (line-break? (z/left* zloc)) (space? zloc)))

(defn- comment-next? [zloc]
  (-> zloc z/next* skip-whitespace comment?))

(defn- comment-next-other-than-line-comment? [zloc]
  (when-let [znext (-> zloc z/next* skip-whitespace)]
    (and (comment? znext) (not (line-comment? znext)))))

(defn- should-indent? [zloc opts]
  (and (line-break? zloc)
       (if (:indent-line-comments? opts)
         (not (comment-next-other-than-line-comment? zloc))
         (not (comment-next? zloc)))))

(defn- should-unindent? [zloc opts]
  (and (indentation? zloc)
       (if (:indent-line-comments? opts)
         (not (comment-next-other-than-line-comment? zloc))
         (not (comment-next? zloc)))))

(defn unindent
  ([form]
   (unindent form {}))
  ([form opts]
   (transform form edit-all #(should-unindent? % opts) z/remove*)))

(def ^:private start-element
  {:meta "^", :meta* "#^", :vector "[",       :map "{"
   :list "(", :eval "#=",  :uneval "#_",      :fn "#("
   :set "#{", :deref "@",  :reader-macro "#", :unquote "~"
   :var "#'", :quote "'",  :syntax-quote "`", :unquote-splicing "~@"
   :namespaced-map "#"})

(defn- prior-line-string [zloc]
  (loop [zloc     zloc
         worklist '()]
    (if-let [p (z/left* zloc)]
      (let [s            (str (n/string (z/node p)))
            new-worklist (cons s worklist)]
        (if-not (includes? s "\n")
          (recur p new-worklist)
          (apply str new-worklist)))
      (if-let [p (z/up* zloc)]
        ;; newline cannot be introduced by start-element
        (recur p (cons (start-element (n/tag (z/node p))) worklist))
        (apply str worklist)))))

(defn- last-line-in-string ^String [^String s]
  (subs s (inc (.lastIndexOf s "\n"))))

(defn- margin [zloc]
  (-> zloc prior-line-string last-line-in-string count))

(defn- whitespace [width]
  (n/whitespace-node (apply str (repeat width " "))))

(defn- coll-indent [zloc]
  (-> zloc z/leftmost* margin))

(defn- uneval? [zloc]
  (= (z/tag zloc) :uneval))

(defn- index-of [zloc]
  (->> (iterate z/left zloc)
       (remove uneval?)
       (take-while identity)
       (count)
       (dec)))

(defn- skip-meta [zloc]
  (if (#{:meta :meta*} (z/tag zloc))
    (-> zloc z/down z/right)
    zloc))

(defn- cursive-two-space-list-indent? [zloc]
  (-> zloc z/leftmost* skip-meta z/tag #{:vector :map :list :set} not))

(defn- zprint-two-space-list-indent? [zloc]
  (-> zloc z/leftmost* z/tag #{:token :list}))

(defn two-space-list-indent? [zloc context]
  (case (:function-arguments-indentation context)
    :community false
    :cursive (cursive-two-space-list-indent? zloc)
    :zprint (zprint-two-space-list-indent? zloc)))

(defn- list-indent [zloc context]
  (if (> (index-of zloc) 1)
    (-> zloc z/leftmost* z/right margin)
    (cond-> (coll-indent zloc)
      (two-space-list-indent? zloc context) inc)))

(def indent-size 2)

(defn- indent-width [zloc]
  (case (z/tag zloc)
    :list indent-size
    :fn   (inc indent-size)))

(defn- remove-namespace [x]
  (if (symbol? x) (symbol (name x)) x))

(defn pattern? [v]
  (instance? #?(:clj Pattern :cljs js/RegExp) v))

#?(:clj
   (defn- top-level-form [zloc]
     (->> zloc
          (iterate z/up)
          (take-while (complement root?))
          last)))

(defn- token? [zloc]
  (= (z/tag zloc) :token))

(defn- ns-token? [zloc]
  (and (token? zloc)
       (= 'ns (z/sexpr zloc))))

(defn- ns-form? [zloc]
  (and (top? zloc)
       (= (z/tag zloc) :list)
       (some-> zloc z/down ns-token?)))

(defn- token-value [zloc]
  (let [zloc (skip-meta zloc)]
    (when (token? zloc) (z/sexpr zloc))))

(defn- reader-conditional? [zloc]
  (and (reader-macro? zloc) (#{"?" "?@"} (-> zloc z/down token-value str))))

(defn- find-next-keyword [zloc]
  (z/find zloc z/right #(n/keyword-node? (z/node %))))

(defn- first-symbol-in-reader-conditional [zloc]
  (when (reader-conditional? zloc)
    (when-let [key-loc (-> zloc z/down z/right z/down find-next-keyword)]
      (when-let [value-loc (-> key-loc z/next skip-meta)]
        (when (token? value-loc)
          (z/sexpr value-loc))))))

(defn- form-symbol [zloc]
  (let [zloc (z/leftmost zloc)]
    (or (token-value zloc)
        (first-symbol-in-reader-conditional zloc))))

(defn- index-matches-top-argument? [zloc depth idx]
  (and (> depth 0)
       (= (inc idx) (index-of (nth (iterate z/up zloc) depth)))))

(defn- qualify-symbol-by-alias-map [possible-sym alias-map]
  (when-let [ns-str (namespace possible-sym)]
    (symbol (get alias-map ns-str ns-str) (name possible-sym))))

(defn- qualify-symbol-by-ns-name [possible-sym ns-name]
  (when ns-name
    (symbol (name ns-name) (name possible-sym))))

(defn- fully-qualified-symbol [possible-sym context]
  (if (symbol? possible-sym)
    (or (qualify-symbol-by-alias-map possible-sym (:alias-map context))
        (qualify-symbol-by-ns-name possible-sym (:ns-name context)))
    possible-sym))

(defn- symbol-matches-key? [sym key]
  (when (symbol? sym)
    (cond
      (symbol? key)  (= key sym)
      (pattern? key) (re-find key (str sym)))))

(defn form-matches-key? [zloc key context]
  (let [possible-sym (form-symbol zloc)]
    (or (symbol-matches-key? (fully-qualified-symbol possible-sym context) key)
        (symbol-matches-key? (remove-namespace possible-sym) key))))

(defn- inner-indent [zloc key depth idx context]
  (let [top (nth (iterate z/up zloc) depth)]
    (when (and (form-matches-key? top key context)
               (or (nil? idx) (index-matches-top-argument? zloc depth idx)))
      (let [zup (z/up zloc)]
        (+ (margin zup) (indent-width zup))))))

(defn- nth-form [zloc n]
  (reduce (fn [z f] (when z (f z)))
          (z/leftmost zloc)
          (repeat n z/right)))

(defn- first-form-in-line? [zloc]
  (and (some? zloc)
       (if-let [zloc (z/left* zloc)]
         (if (space? zloc)
           (recur zloc)
           (or (z/linebreak? zloc) (comment? zloc)))
         true)))

(defn- block-indent [zloc key idx context]
  (when (form-matches-key? zloc key context)
    (let [zloc-after-idx (some-> zloc (nth-form (inc idx)))]
      (if (and (or (nil? zloc-after-idx) (first-form-in-line? zloc-after-idx))
               (> (index-of zloc) idx))
        (inner-indent zloc key 0 nil context)
        (list-indent zloc context)))))

(def default-indents
  (merge (read-resource "cljfmt/indents/clojure.clj")
         (read-resource "cljfmt/indents/compojure.clj")
         (read-resource "cljfmt/indents/fuzzy.clj")))

(def default-aligned-forms
  (read-resource "cljfmt/aligned_forms/clojure.clj"))

(def default-options
  {:alias-map                             {}
   :align-binding-columns?                false
   :align-map-columns?                    false
   :aligned-forms                         default-aligned-forms
   :extra-aligned-forms                   {}
   :extra-indents                         {}
   :function-arguments-indentation        :community
   :indent-line-comments?                 false
   :indentation?                          true
   :indents                               default-indents
   :insert-missing-whitespace?            true
   :remove-consecutive-blank-lines?       true
   :remove-multiple-non-indenting-spaces? false
   :remove-surrounding-whitespace?        true
   :remove-trailing-whitespace?           true
   :sort-ns-references?                   false
   :split-keypairs-over-multiple-lines?   false})

(defmulti ^:private indenter-fn
  (fn [_sym _context [type & _args]] type))

(defmethod indenter-fn :inner [sym context [_ depth idx]]
  (fn [zloc] (inner-indent zloc sym depth idx context)))

(defmethod indenter-fn :block [sym context [_ idx]]
  (fn [zloc] (block-indent zloc sym idx context)))

(defmethod indenter-fn :default [sym context [_]]
  (fn [zloc]
    (when (form-matches-key? zloc sym context)
      (list-indent zloc context))))

(defn- make-indenter [[key opts] context]
  (apply some-fn (map (partial indenter-fn key context) opts)))

(defn- indent-order [[key specs]]
  (let [get-depth (fn [[type depth]] (if (= type :inner) depth 0))
        max-depth (transduce (map get-depth) max 0 specs)
        key-order  (cond
                     (qualified-symbol? key) 0
                     (simple-symbol? key)    1
                     (pattern? key)          2)]
    [(- max-depth) key-order (str key)]))

(defn- custom-indent [zloc indents context]
  (if (empty? indents)
    (list-indent zloc context)
    (let [indenter (->> indents
                        (map #(make-indenter % context))
                        (apply some-fn))]
      (or (indenter zloc)
          (list-indent zloc context)))))

(defn- indent-amount [zloc indents context]
  (let [tag (-> zloc z/up z/tag)
        gp  (-> zloc z/up z/up)]
    (cond
      (reader-conditional? gp) (coll-indent zloc)
      (#{:list :fn} tag)       (custom-indent zloc indents context)
      (= :meta tag)            (indent-amount (z/up zloc) indents context)
      :else                    (coll-indent zloc))))

(defn- indent-line [zloc indents context]
  (let [width (indent-amount zloc indents context)]
    (if (> width 0)
      (z/insert-right* zloc (whitespace width))
      zloc)))

(defn- find-namespace [zloc]
  (some-> zloc root z/down (z/find z/right ns-form?) z/down z/next z/sexpr))

(defn indent
  ([form]
   (indent form default-indents {}))
  ([form indents]
   (indent form indents {}))
  ([form indents alias-map]
   (indent form indents alias-map default-options))
  ([form indents alias-map opts]
   (let [ns-name (find-namespace (z/of-node form))
         sorted-indents (sort-by indent-order indents)
         context (merge (select-keys opts [:function-arguments-indentation])
                        {:alias-map alias-map
                         :ns-name ns-name})]
     (transform form edit-all #(should-indent? % opts)
                #(indent-line % sorted-indents context)))))

(defn- map-key? [zloc]
  (and (z/map? (z/up zloc))
       (even? (index-of zloc))
       (not (uneval? zloc))
       (not (z/whitespace-or-comment? zloc))))

(defn- preceded-by-line-break? [zloc]
  (loop [previous (z/left* zloc)]
    (cond
      (line-break? previous)
      true
      (z/whitespace-or-comment? previous)
      (recur (z/left* previous)))))

(defn- map-key-without-line-break? [zloc]
  (and (map-key? zloc) (not (preceded-by-line-break? zloc))))

(defn- insert-newline-left [zloc]
  (z/insert-left* zloc (n/newlines 1)))

(defn split-keypairs-over-multiple-lines [form]
  (transform form edit-all map-key-without-line-break? insert-newline-left))

(defn reindent
  ([form]
   (indent (unindent form)))
  ([form indents]
   (indent (unindent form) indents))
  ([form indents alias-map]
   (indent (unindent form) indents alias-map))
  ([form indents alias-map opts]
   (indent (unindent form opts) indents alias-map opts)))

(defn final? [zloc]
  (and (nil? (z/right* zloc)) (root? (z/up* zloc))))

(defn- trailing-whitespace? [zloc]
  (and (space? zloc)
       (or (z/linebreak? (z/right* zloc)) (final? zloc))))

(defn remove-trailing-whitespace [form]
  (transform form edit-all trailing-whitespace? z/remove*))

(defn- replace-with-one-space [zloc]
  (z/replace* zloc (whitespace 1)))

(defn- non-indenting-whitespace? [zloc]
  (and (space? zloc) (not (indentation? zloc))))

(defn remove-multiple-non-indenting-spaces [form]
  (transform form edit-all non-indenting-whitespace? replace-with-one-space))

(def ^:private ns-reference-symbols
  #{:import :require :require-macros :use})

(defn- ns-reference? [zloc]
  (and (z/list? zloc)
       (some-> zloc z/up ns-form?)
       (-> zloc z/sexpr first ns-reference-symbols)))

(defn- re-indexes [re s]
  (let [matcher    #?(:clj  (re-matcher re s)
                      :cljs (js/RegExp. (.-source re) "g"))
        next-match #?(:clj  #(when (.find matcher)
                               [(.start matcher) (.end matcher)])
                      :cljs #(when-let [result (.exec matcher s)]
                               [(.-index result) (.-lastIndex matcher)]))]
    (take-while some? (repeatedly next-match))))

(defn- re-seq-matcher [re charmap coll]
  {:pre (every? charmap coll)}
  (let [s (apply str (map charmap coll))
        v (vec coll)]
    (for [[start end] (re-indexes re s)]
      {:value (subvec v start end)
       :start start
       :end   end})))

(defn- find-elements-with-comments [nodes]
  (re-seq-matcher #"(CNS*)*E(S*C)?"
                  #(case (n/tag %)
                     (:whitespace :comma) \S
                     :comment \C
                     :newline \N
                     \E)
                  nodes))

(defn- splice-into [coll splices]
  (letfn [(splice [v i splices]
            (when-let [[{:keys [value start end]} & splices] (seq splices)]
              (lazy-cat (subvec v i start) value (splice v end splices))))]
    (splice (vec coll) 0 splices)))

(defn- add-newlines-after-comments [nodes]
  (mapcat #(if (n/comment? %) [% (n/newlines 1)] [%]) nodes))

(defn- remove-newlines-after-comments [nodes]
  (mapcat #(when-not (and %1 (n/comment? %1) (n/linebreak? %2)) [%2])
          (cons nil nodes)
          nodes))

(defn- sort-node-arguments-by [f nodes]
  (let [nodes  (add-newlines-after-comments nodes)
        args   (rest (find-elements-with-comments nodes))
        sorted (sort-by f (map :value args))]
    (->> sorted
         (map #(assoc %1 :value %2) args)
         (splice-into nodes)
         (remove-newlines-after-comments))))

(defn- update-children [zloc f]
  (let [node (z/node zloc)]
    (z/replace zloc (n/replace-children node (f (n/children node))))))

(defn- nodes-string [nodes]
  (apply str (map n/string nodes)))

(defn- remove-node-metadata [nodes]
  (mapcat #(if (= (n/tag %) :meta)
             (rest (n/children %))
             [%])
          nodes))

(defn- node-sort-string [nodes]
  (-> (remove (some-fn n/comment? n/whitespace?) nodes)
      (remove-node-metadata)
      (nodes-string)
      (str/replace #"[\[\]\(\)\{\}]" "")
      (str/trim)))

(defn sort-arguments [zloc]
  (update-children zloc #(sort-node-arguments-by node-sort-string %)))

(defn sort-ns-references [form]
  (transform form edit-all ns-reference? sort-arguments))

(defn- skip-to-linebreak-or-element [zloc]
  (z/skip z/right* (some-fn space? comma?) zloc))

(defn- reduce-columns [zloc f init]
  (loop [zloc zloc, col 0, acc init]
    (if-some [zloc (skip-to-linebreak-or-element zloc)]
      (if (line-break? zloc)
        (recur (z/right* zloc) 0 acc)
        (recur (z/right* zloc) (inc col) (f zloc col acc)))
      acc)))

(defn- count-columns [zloc]
  (inc (reduce-columns zloc #(max %2 %3) 0)))

(defn- trailing-commas [zloc]
  (let [right (z/right* zloc)]
    (if (and right (comma? right))
      (-> right z/node n/string)
      "")))

(defn- node-end-position [zloc]
  (let [lines (str (prior-line-string zloc)
                   (n/string (z/node zloc))
                   (trailing-commas zloc))]
    (transduce (comp (remove #(str/starts-with? % ";"))
                     (map count))
               max 0 (str/split lines #"\r?\n"))))

(defn- max-column-end-position [zloc col]
  (reduce-columns zloc
                  (fn [zloc c max-pos]
                    (if (= c col)
                      (max max-pos (node-end-position zloc))
                      max-pos))
                  0))

(defn- update-space-left [zloc delta]
  (let [left (z/left* zloc)]
    (cond
      (space? left) (let [n (-> left z/node n/string count)]
                      (z/right* (z/replace* left (n/spaces (+ n delta)))))
      (pos? delta)  (z/insert-space-left zloc delta)
      :else         zloc)))

(defn- nil-if-end [zloc]
  (when (and zloc (not (z/end? zloc))) zloc))

(defn- skip-to-next-line [zloc]
  (->> zloc (z/skip z/next* (complement line-break?)) z/next nil-if-end))

(defn- pad-inner-node [zloc padding]
  (if-some [zloc (z/down zloc)]
    (loop [zloc zloc]
      (if-some [zloc (skip-to-next-line zloc)]
        (recur (update-space-left zloc padding))
        zloc))
    zloc))

(defn- pad-node [zloc start-position]
  (let [padding (- start-position (margin zloc))
        zloc    (update-space-left zloc padding)]
    (z/subedit-> zloc (pad-inner-node padding))))

(defn- edit-column [zloc column f]
  (loop [zloc zloc, col 0]
    (if-some [zloc (skip-to-linebreak-or-element zloc)]
      (let [zloc (if (and (= col column) (not (line-break? zloc)))
                   (f zloc)
                   zloc)
            col  (if (line-break? zloc) 0 (inc col))]
        (if-some [zloc (z/right* zloc)]
          (recur zloc col)
          zloc))
      zloc)))

(defn- align-one-column [zloc col]
  (if-some [zloc (z/down zloc)]
    (let [start-position (inc (max-column-end-position zloc (dec col)))]
      (z/up (edit-column zloc col #(pad-node % start-position))))
    zloc))

(defn- align-columns [zloc]
  (reduce align-one-column zloc (-> zloc z/down count-columns range rest)))

(defn align-map-columns [form]
  (transform form edit-all z/map? align-columns))

(defn- aligned-form? [zloc aligned-forms context]
  (and (z/list? (z/up zloc))
       (some (fn [[k indexes]]
               (and (form-matches-key? zloc k context)
                    (contains? (set indexes) (dec (index-of zloc)))))
             aligned-forms)))

(defn align-form-columns [form aligned-forms alias-map]
  (let [ns-name  (find-namespace (z/of-node form))
        context  {:alias-map alias-map, :ns-name ns-name}
        aligned? #(aligned-form? % aligned-forms context)]
    (transform form edit-all aligned? align-columns)))

(defn realign-form
  "Realign a rewrite-clj form such that the columns line up."
  [form]
  (-> form z/of-node align-columns z/root))

#?(:clj
   (defn- ns-require-form? [zloc]
     (and (some-> zloc top-level-form ns-form?)
          (some-> zloc z/child-sexprs first (= :require)))))

#?(:clj
   (defn- as-keyword? [zloc]
     (and (= :token (z/tag zloc))
          (= :as (z/sexpr zloc)))))

#?(:clj
   (defn- symbol-node? [zloc]
     (some-> zloc z/node n/symbol-node?)))

#?(:clj
   (defn- leftmost-symbol [zloc]
     (some-> zloc z/leftmost (z/find (comp symbol-node? skip-meta)))))

#?(:clj
   (defn- as-zloc->alias-mapping [as-zloc]
     (let [alias             (some-> as-zloc z/right z/sexpr)
           current-namespace (some-> as-zloc leftmost-symbol z/sexpr)
           grandparent-node  (some-> as-zloc z/up z/up)
           parent-namespace  (when-not (ns-require-form? grandparent-node)
                               (when (or (z/vector? grandparent-node)
                                         (z/list? grandparent-node))
                                 (first (z/child-sexprs grandparent-node))))]
       (when (and (symbol? alias) (symbol? current-namespace))
         {(str alias) (if parent-namespace
                        (format "%s.%s" parent-namespace current-namespace)
                        (str current-namespace))}))))

#?(:clj
   (defn- alias-map-for-form [form]
     (when-let [req-zloc (-> form z/of-node (z/find z/next ns-require-form?))]
       (->> (find-all req-zloc as-keyword?)
            (map as-zloc->alias-mapping)
            (apply merge)))))

(defn- stringify-map [m]
  (into {} (map (fn [[k v]] [(str k) (str v)])) m))

(defn reformat-form
  "Reformats a rewrite-clj form data structure. Accepts a map of
  [formatting options][1]. See also: [[reformat-string]].
  
  [1]: https://github.com/weavejester/cljfmt#formatting-options"
  ([form]
   (reformat-form form {}))
  ([form options]
   (let [opts      (merge default-options options)
         indents   (merge (:indents opts) (:extra-indents opts))
         aligned   (merge (:aligned-forms opts) (:extra-aligned-forms opts))
         alias-map #?(:clj  (merge (alias-map-for-form form)
                                   (stringify-map (:alias-map opts)))
                      :cljs (stringify-map (:alias-map opts)))]
     (-> form
         (cond-> (:sort-ns-references? opts)
           sort-ns-references)
         (cond-> (:split-keypairs-over-multiple-lines? opts)
           split-keypairs-over-multiple-lines)
         (cond-> (:remove-consecutive-blank-lines? opts)
           remove-consecutive-blank-lines)
         (cond-> (:remove-surrounding-whitespace? opts)
           remove-surrounding-whitespace)
         (cond-> (:insert-missing-whitespace? opts)
           insert-missing-whitespace)
         (cond-> (:remove-multiple-non-indenting-spaces? opts)
           remove-multiple-non-indenting-spaces)
         (cond-> (:indentation? opts)
           (reindent indents alias-map opts))
         (cond-> (:align-map-columns? opts)
           align-map-columns)
         (cond-> (:align-form-columns? opts)
           (align-form-columns aligned alias-map))
         (cond-> (:remove-trailing-whitespace? opts)
           remove-trailing-whitespace)))))

(defn reformat-string
  "Reformat a string. Accepts a map of [formatting options][1].

  [1]: https://github.com/weavejester/cljfmt#formatting-options"
  ([form-string]
   (reformat-string form-string {}))
  ([form-string options]
   (-> (p/parse-string-all form-string)
       (reformat-form options)
       (n/string))))

(def default-line-separator
  #?(:clj (System/lineSeparator) :cljs \newline))

(defn normalize-newlines [s]
  (str/replace s #"\r\n" "\n"))

(defn replace-newlines [s sep]
  (str/replace s #"\n" sep))

(defn find-line-separator [s]
  (or (re-find #"\r?\n" s) default-line-separator))

(defn wrap-normalize-newlines [f]
  (fn [s]
    (let [sep (find-line-separator s)]
      (-> s normalize-newlines f (replace-newlines sep)))))
