(ns vlaaad.reveal.pro.fs
  "© 2021 Vladislav Protsenko. All rights reserved."
  (:require [vlaaad.reveal.action :as action]
            [vlaaad.reveal.view :as view]
            [vlaaad.reveal.output-panel :as output-panel]
            [vlaaad.reveal.event :as event]
            [vlaaad.reveal.fx :as rfx]
            [vlaaad.reveal.stream :as stream]
            [clojure.string :as str])
  (:import [java.nio.file Path Paths Files LinkOption FileSystems]
           [java.io File]
           [java.net URL URI]
           [java.nio.charset StandardCharsets]
           [java.util.stream Collectors]
           [java.nio.file.spi FileSystemProvider]))

(stream/defstream Path [x]
  (stream/horizontal
    (stream/raw-string "#reveal/path" {:fill :object})
    stream/separator
    (stream/stream (str x))))

(defn- on-path [x f]
  (cond
    (instance? Path x)
    (f x)

    (and (instance? URI x)
         (let [scheme (.getScheme ^URI x)]
           (some #(= scheme (.getScheme ^FileSystemProvider %))
                 (FileSystemProvider/installedProviders))))
    (recur (Paths/get ^URI x) f)

    (instance? File x)
    (recur (.toPath ^File x) f)

    (instance? URL x)
    (recur (.toURI ^URL x) f)))

(action/defaction ::action/fs:attributes [x]
  (on-path x
    (fn [^Path x]
      #(stream/vertically
         (for [[k v] (->> (.getFileSystem x)
                          (.supportedFileAttributeViews)
                          (mapcat
                            (fn [view]
                              {view (into {} (Files/readAttributes
                                               x
                                               (str view ":*")
                                               ^"[Ljava.nio.file.LinkOption;" (into-array LinkOption [])))}))
                          (sort-by (comp not #{"basic"} key))
                          (map val)
                          (apply merge)
                          sort)]
           (stream/horizontal
             (stream/as k (stream/raw-string k {:fill :symbol}))
             stream/separator
             (stream/stream v)))))))

(action/defaction ::action/fs:parent [x]
  (on-path x
    (fn [^Path x]
      #(.getParent (.toAbsolutePath x)))))

(defn- has-children? [^Path path]
  (or (Files/isDirectory path (into-array LinkOption []))
      (let [s (str path)]
        (or (str/ends-with? s ".zip")
            (str/ends-with? s ".jar")))))

(action/defaction ::action/fs:tree [x]
  (on-path x
    (fn [^Path p]
      (when (has-children? p)
        (fn []
          {:fx/type view/tree-view
           :render #(-> % :name (stream/raw-string {:fill :symbol}))
           :valuate :path
           :root {:name (str p)
                  :path p}
           :branch? (comp has-children? :path)
           :children (fn [{:keys [^Path path]}]
                       (if (Files/isDirectory path (into-array LinkOption []))
                         (map (fn [^Path p]
                                {:name (str (.getFileName p))
                                 :path p})
                              (.collect (Files/list path) (Collectors/toList)))
                         (let [loader nil]
                           (->> (FileSystems/newFileSystem ^Path path ^ClassLoader loader)
                                .getRootDirectories
                                (mapcat
                                  (fn [path]
                                    (.collect (Files/list path) (Collectors/toList))))
                                (map (fn [^Path p]
                                       {:name (str (.getFileName p))
                                        :path p}))))))})))))

(defn- process-buffered-reader! [id path handler]
  (handler {::event/type ::view/create-view-state :id id :state (output-panel/make {:autoscroll false})})
  (let [*running (volatile! true)
        add-lines! #(handler {::event/type ::output-panel/on-add-lines :id id :fx/event %})
        xform (comp stream/stream-xf
                    (partition-all 128)
                    (take-while (fn [_] @*running)))
        reader (Files/newBufferedReader path StandardCharsets/UTF_8)
        f (event/daemon-future
            (try
              (loop []
                (when @*running
                    (when-let [str (.readLine reader)]
                      (view/runduce! xform add-lines! (stream/as str (stream/raw-string str {:fill :symbol})))
                      (recur))))
              (catch Exception e (view/runduce! xform add-lines! e))))]
    #(do
       (handler {::event/type ::view/dispose-state :id id})
       (future-cancel f)
       (vreset! *running false)
       (.close reader))))

(defn- path-reader-view [{:keys [path]}]
  {:fx/type rfx/ext-with-process
   :start process-buffered-reader!
   :args path
   :desc {:fx/type output-panel/view}})

(action/defaction ::action/fs:read-lines [x]
  (on-path x
    (fn [x]
      (when-not (Files/isDirectory x (into-array LinkOption []))
        (constantly {:fx/type path-reader-view :path x})))))