(ns vlaaad.reveal.pro.system
  (:require [vlaaad.reveal.view :as view]
            [vlaaad.reveal.stream :as stream]
            [vlaaad.reveal.event :as event]
            [vlaaad.reveal.ui :as ui]
            [vlaaad.reveal.ns :as ns]
            [vlaaad.reveal.action-popup :as action-popup]
            [vlaaad.reveal.style :as style]
            [cljfx.fx.h-box :as fx.h-box]
            [cljfx.fx.label :as fx.label]
            [cljfx.fx.button :as fx.button]
            [cljfx.fx.region :as fx.region])
  (:import [clojure.lang Atom Agent Ref Var]))

(defn- system-view [{:keys [description running system switch]}]
  {:fx/type fx.h-box/lifecycle
   :padding {:left style/default-padding}
   :children [{:fx/type action-popup/ext
               :h-box/hgrow :always
               :value system
               :desc {:fx/type fx.label/lifecycle
                      :style-class ["label" "reveal-system-label"]
                      :pseudo-classes #{(if running :running :stopped)}
                      :max-width ##Inf
                      :max-height ##Inf
                      :text (str
                              (if running
                                "running"
                                "stopped")
                              (when (and running description)
                                (str " (" description ")")))}}
              {:fx/type fx.button/lifecycle
               :graphic {:fx/type fx.region/lifecycle
                         :style-class "reveal-system-icon"
                         :pseudo-classes #{(if running :stop :start)}
                         :min-width :use-pref-size
                         :max-width :use-pref-size
                         :min-height :use-pref-size
                         :max-height :use-pref-size}
               :on-action switch}]})

(defn system-watcher [{:keys [ref describe switch running]}]
  (let [switch (fn [_] (event/daemon-future
                         (try
                           (switch ref)
                           (catch Exception e
                             (ui/inspect
                               (stream/override-style
                                 (stream/stream e)
                                 assoc :fill :error))))))]
    {:fx/type view/observable-view
     :ref ref
     :fn (fn [system]
           {:fx/type system-view
            :system system
            :description (describe system)
            :running (running system)
            :switch switch})}))

(defn- swap-once [ref f & args]
  (condp instance? ref
    Atom (let [old @ref
               new (apply f old args)]
           (when-not (compare-and-set! ref old new)
             (throw (ex-info "Switched system state, but ref was already different"
                             {:ref ref
                              :old old
                              :new new}))))
    Agent (apply send-via event/daemon-executor ref f args)
    Ref (let [tried (atom false)]
          (dosync
            (alter ref (fn [v]
                         (if (compare-and-set! tried false true)
                           (apply f v args)
                           ;; TODO throw instead?
                           v)))))
    Var (apply alter-var-root ref f args)))

(defn- not-on-the-classpath-view [lib props]
  {:fx/type view/derefable
   :derefable (delay (throw (ex-info (str lib " is not on the classpath")
                                     (assoc props :lib lib))))})

(def integrant-view
  (or (ns/when-exists integrant.core
        (let [init (requiring-resolve 'integrant.core/init)
              prep (requiring-resolve 'integrant.core/prep)
              halt (requiring-resolve 'integrant.core/halt!)
              default-switch (fn [sys config ks]
                               (if sys
                                 (do (halt sys) nil)
                                 (try
                                   (-> config prep (init (or ks (keys config))))
                                   (catch Exception ex
                                     (when-let [sys (:system (ex-data ex))]
                                       (try
                                         (halt sys)
                                         (catch Exception halt-ex
                                           (throw (ex-info "Failed to init, then failed to halt"
                                                           {:init-exception ex}
                                                           halt-ex)))))
                                     (throw ex)))))]
          (fn integrant-view [{:keys [ref config keys describe running switch]
                               :or {describe count
                                    running some?}}]
            {:fx/type system-watcher
             :ref ref
             :switch (or switch #(swap-once % default-switch config keys))
             :describe describe
             :running running})))
      #(not-on-the-classpath-view 'integrant.core %)))

(defn integrant-sticker [opts]
  (ui/sticker (merge {:fx/type integrant-view}
                     (select-keys opts [:ref :switch :config :keys :describe :running]))
              (merge {:title "integrant system"
                      :default-size {:width 154 :height 46}}
                     (dissoc opts :ref :switch :config :keys :describe :running))))

(def integrant-repl-view
  (or (ns/when-exists integrant.repl
        (let [sys (requiring-resolve 'integrant.repl.state/system)
              preparer (requiring-resolve 'integrant.repl.state/preparer)]
          (fn integrant-repl-view [props]
            {:fx/type view/observable-view
             :ref preparer
             :fn (fn [mk-config]
                   (assoc props
                     :fx/type integrant-view
                     :ref sys
                     :config (if mk-config (mk-config) {})))})))
      #(not-on-the-classpath-view 'integrant.repl %)))

(defn integrant-repl-sticker [opts]
  (ui/sticker (merge {:fx/type integrant-repl-view}
                     (select-keys opts [:switch :keys :describe :running]))
              (merge {:title "integrant repl system"
                      :default-size {:width 154 :height 46}}
                     (dissoc opts :switch :keys :describe :running))))

(def mount-view
  (or (ns/when-exists mount.core
        (let [sys @(requiring-resolve 'mount.core/running)
              start (requiring-resolve 'mount.core/start)
              stop (requiring-resolve 'mount.core/stop)
              running? (comp pos? count)
              default-switch (fn [ref states]
                               (if (running? @ref)
                                 (stop)
                                 (apply start states)))]
          (fn mount-view [{:keys [states describe switch]
                           :or {describe count}}]
            {:fx/type system-watcher
             :ref sys
             :switch (or switch #(default-switch % states))
             :describe describe
             :running running?})))
      #(not-on-the-classpath-view 'mount.core %)))

(defn mount-sticker [opts]
  (ui/sticker (merge {:fx/type mount-view}
                     (select-keys opts [:states :describe :switch]))
              (merge {:title "mount system"
                      :default-size {:width 154 :height 46}}
                     (dissoc opts :states :describe :switch))))

(def component-view
  (or (ns/when-exists com.stuartsierra.component
        (let [start (requiring-resolve 'com.stuartsierra.component/start)
              stop (requiring-resolve 'com.stuartsierra.component/stop)
              default-describe #(if (map? %) (count %) nil)
              default-switch (fn [ref running]
                               (swap-once ref #(if (running %) (stop %) (start %))))]
          (fn component-view [{:keys [ref switch describe running]
                               :or {describe default-describe}}]
            {:fx/type system-watcher
             :ref ref
             :switch (or switch #(default-switch % running))
             :describe describe
             :running running})))
      #(not-on-the-classpath-view 'com.stuartsierra.component %)))

(defn component-sticker [opts]
  (ui/sticker (merge {:fx/type component-view}
                     (select-keys opts [:ref :switch :describe :running]))
              (merge {:title "component system"
                      :default-size {:width 154 :height 46}}
                     (dissoc opts :ref :switch :describe :running))))
