(ns vlaaad.reveal.pro.form.vignette
  "© 2021 Vladislav Protsenko. All rights reserved."
  (:require [vlaaad.reveal.pro.form.impl :as impl]
            [vlaaad.reveal.event :as event]
            [clojure.main :as m]
            [cljfx.api :as fx]
            [cljfx.coerce :as fx.coerce]
            [vlaaad.reveal.fx :as rfx]
            [cljfx.lifecycle :as fx.lifecycle]
            [vlaaad.reveal.popup :as popup]
            [vlaaad.reveal.style :as style]
            [vlaaad.reveal.font :as font]
            [cljfx.fx.node :as fx.node]
            [clojure.string :as str]
            [cljfx.fx.label :as fx.label]
            [cljfx.fx.progress-indicator :as fx.progress-indicator]
            [cljfx.fx.v-box :as fx.v-box]
            [cljfx.fx.h-box :as fx.h-box]
            [cljfx.fx.region :as fx.region]
            [cljfx.fx.scroll-pane :as fx.scroll-pane])
  (:import [javafx.scene.input KeyEvent KeyCode ContextMenuEvent KeyCombination]
           [javafx.scene Node]
           [javafx.event Event]))

(defn- hide-popup [state]
  (dissoc state :node :option-future :option-error))

(defmethod event/handle ::hide-popup [{:keys [on-state-changed]}]
  (event/handle (assoc on-state-changed :fn hide-popup)))

(defn- move-selected-index [state options direction]
  (let [max (dec (count options))
        i (direction (:option-index state 0))]
    (-> state
        (assoc :option-index (cond
                               (neg? i) max
                               (< max i) 0
                               :else i))
        (dissoc :option-future :option-error))))

(defmethod event/handle ::select-option [{:keys [options option-index on-state-changed]}]
  (let [fn-or-event (:invoke (options option-index))]
    (comp
      (event/handle (assoc on-state-changed :fn #(assoc % :option-index option-index)))
      (if (::event/type fn-or-event)
        (comp (event/handle fn-or-event)
              (event/handle {::event/type ::hide-popup
                             :on-state-changed on-state-changed}))
        (let [f (event/daemon-future (fn-or-event))]
          (event/handle (assoc on-state-changed :fn #(-> %
                                                         (assoc :option-future f)
                                                         (dissoc :option-error)))))))))

(defn- matching-option-index [options event]
  (->> options
       (keep-indexed
         (fn [i option]
           (when (and (:shortcut option)
                      (-> option
                          :shortcut
                          ^KeyCombination fx.coerce/key-combination
                          (.match event)))
             i)))
       first))

(defn- key-matcher [combo]
  (let [^KeyCombination c (fx.coerce/key-combination combo)]
    (fn match [e]
      (.match c e))))

(def ^:private up? (key-matcher [:up]))
(def ^:private down? (key-matcher [:down]))
(def ^:private enter? (key-matcher [:enter]))
(def ^:private space? (key-matcher [:space]))
(def ^:private f1? (key-matcher [:f1]))
(def ^:private ctrl-space? (key-matcher [:ctrl :space]))

(defmethod event/handle ::handle-popup-key-press
  [{:keys [^KeyEvent fx/event on-state-changed options option-index]}]
  (condp = (.getCode event)
    KeyCode/ESCAPE (event/handle (assoc on-state-changed :fn hide-popup))
    KeyCode/TAB (event/handle (assoc on-state-changed :fn hide-popup))
    (if (pos? (count options))
      (cond
        (up? event) (event/handle (assoc on-state-changed :fn #(move-selected-index % options dec)))
        (down? event) (event/handle (assoc on-state-changed :fn #(move-selected-index % options inc)))
        (enter? event) (event/handle {::event/type ::select-option
                                      :options options
                                      :option-index option-index
                                      :on-state-changed on-state-changed})
        :else (if-let [i (matching-option-index options event)]
                (event/handle {::event/type ::select-option
                               :options options
                               :option-index i
                               :on-state-changed on-state-changed})
                identity))
      identity)))

(defmethod event/handle ::commit-option [{:keys [on-state-changed on-edit value form]}]
  (let [{:keys [editor]} form
        new-edit (impl/edit editor value)]
    (comp
      (event/handle (assoc on-edit :fn (constantly new-edit)))
      (event/handle (assoc on-state-changed :fn #(dissoc % :option-future))))))

(defn- await-option-future! [_ {:keys [option-future on-state-changed on-edit form]} handler]
  (let [cancelled (volatile! false)
        f (event/daemon-future
            (try
              (handler
                {::event/type ::commit-option
                 :form form
                 :on-state-changed on-state-changed
                 :on-edit on-edit
                 :value @option-future})
              (catch Exception e
                (when-not @cancelled
                  (let [error (-> e Throwable->map m/ex-triage (:clojure.error/cause "invalid option"))]
                    (handler
                      (assoc on-state-changed
                        :fn #(-> %
                                 (dissoc :option-future)
                                 (assoc :option-error error)))))))))]
    #(do
       (vreset! cancelled true)
       (future-cancel f))))

(defn- shortcut-view [{:keys [shortcut]}]
  {:fx/type fx.label/lifecycle
   :style-class "reveal-form-option-shortcut"
   :text (.getDisplayText ^KeyCombination (fx.coerce/key-combination shortcut))})

(defn- popup-view [{:keys [edit form state on-state-changed]}]
  (let [{:keys [description options explain]} form
        {:keys [option-index option-future option-error ^Node node]
         :or {option-index 0}} state
        error (or option-error (impl/error (impl/validate explain edit)))]
    {:fx/type popup/view
     :window (.getWindow (.getScene node))
     :bounds (.localToScreen node (.getBoundsInLocal node))
     :on-cancel {::event/type ::hide-popup :on-state-changed on-state-changed}
     :position :top
     :alignment :left
     :desc {:fx/type fx.v-box/lifecycle
            :focus-traversable true
            :on-key-pressed {::event/type ::handle-popup-key-press
                             :option-index option-index
                             :options options
                             :on-state-changed on-state-changed}
            :padding style/default-padding
            :spacing style/default-padding
            :children
            (cond-> []
              (pos? (count options))
              (conj {:fx/type fx.v-box/lifecycle
                     :fx/key :options
                     :children
                     (for [[i {:keys [label shortcut]}] (map-indexed vector options)
                           :let [selected (= i option-index)]]
                       {:fx/type fx.h-box/lifecycle
                        :spacing style/default-padding
                        :style-class "reveal-form-option"
                        :pseudo-classes (if selected #{:selected} #{})
                        :on-mouse-clicked {::event/type ::select-option
                                           :options options
                                           :option-index i
                                           :on-state-changed on-state-changed}
                        :children (cond-> [{:fx/type fx.label/lifecycle
                                            :style-class "reveal-form-option-label"
                                            :content-display :right
                                            :wrap-text true
                                            :text (str label)}]
                                    (and selected option-future)
                                    (conj {:fx/type fx.progress-indicator/lifecycle
                                           :style {:-fx-progress-color (style/color :symbol)}
                                           :max-width (font/line-height)
                                           :max-height (font/line-height)})

                                    shortcut
                                    (conj {:fx/type fx.region/lifecycle
                                           :h-box/hgrow :always}
                                          {:fx/type shortcut-view
                                           :shortcut shortcut}))})})
              (and (pos? (count options)) (or description error))
              (conj {:fx/type fx.region/lifecycle :style-class "reveal-form-separator"})
              description
              (conj {:fx/type fx.v-box/lifecycle
                     :fx/key :description
                     :children [{:fx/type fx.label/lifecycle
                                 :style-class "reveal-form-popup-header"
                                 :text "description:"}
                                {:fx/type fx.scroll-pane/lifecycle
                                 :min-height 0
                                 :max-height (* 12 (font/line-height))
                                 :hbar-policy :never
                                 :content {:fx/type fx.label/lifecycle
                                           :min-width :use-pref-size
                                           :min-height :use-pref-size
                                           :text description}}]})
              error
              (conj {:fx/type fx.v-box/lifecycle
                     :fx/key :error
                     :children [{:fx/type fx.label/lifecycle
                                 :style-class "reveal-form-popup-header"
                                 :text "error:"}
                                {:fx/type fx.scroll-pane/lifecycle
                                 :min-height 0
                                 :max-height (* 12 (font/line-height))
                                 :hbar-policy :never
                                 :content {:fx/type fx.label/lifecycle
                                           :wrap-text true
                                           :style-class "reveal-form-error"
                                           :text error}}]}))}}))

(def ext-with-node-props (fx/make-ext-with-props fx.node/props))

(defmethod event/handle ::filter-popup-wrapper-events
  [{:keys [main on-state-changed options ^Event fx/event]}]
  (cond
    (or (instance? ContextMenuEvent event)
        (and (instance? KeyEvent event)
             (= (.getEventType event) KeyEvent/KEY_PRESSED)
             (or (f1? event)
                 (ctrl-space? event)))
        (and main
             (instance? KeyEvent event)
             (= (.getEventType event) KeyEvent/KEY_PRESSED)
             (or (space? event) (enter? event))))
    (do (.consume event)
        (event/handle (assoc on-state-changed :fn #(assoc % :node (.getTarget event)))))

    (and (instance? KeyEvent event)
         (= (.getEventType event) KeyEvent/KEY_PRESSED))
    (if-let [i (matching-option-index options event)]
      (do
        (.consume event)
        (event/handle {::event/type ::select-option
                       :options options
                       :option-index i
                       :on-state-changed on-state-changed}))
      identity)

    :else
    identity))

(defn- popup-wrapper-impl [{:keys [edit on-edit form desc state on-state-changed main]
                            :or {main false}}]
  (let [{:keys [explain label options]} form
        result (impl/validate explain edit)
        {:keys [node option-future]} state]
    {:fx/type fx/ext-let-refs
     :refs (cond-> {}
             node
             (assoc ::popup {:fx/type popup-view
                             :state state
                             :on-state-changed on-state-changed
                             :edit edit
                             :on-edit on-edit
                             :form form})
             option-future
             (assoc ::await {:fx/type rfx/ext-with-process
                             :args {:option-future option-future
                                    :on-state-changed on-state-changed
                                    :on-edit on-edit
                                    :form form}
                             :start await-option-future!
                             :desc {:fx/type fx.lifecycle/scalar}}))
     :desc
     {:fx/type fx.h-box/lifecycle
      :spacing (font/char-width)
      :children [{:fx/type ext-with-node-props
                  :props (cond-> {:event-filter {::event/type ::filter-popup-wrapper-events
                                                 :main main
                                                 :options options
                                                 :on-state-changed on-state-changed}}
                           node (assoc :pseudo-classes #{:passive}))
                  :desc desc}
                 {:fx/type fx.label/lifecycle
                  :style-class "reveal-form-hint"
                  :text (str/replace (str label) \newline \space)}
                 {:fx/type fx.h-box/lifecycle
                  :spacing (font/char-width)
                  :children (if (impl/error result)
                              [{:fx/type fx.label/lifecycle
                                :style-class "reveal-form-hint"
                                :pseudo-classes #{:separator}
                                :text "-"}
                               {:fx/type fx.label/lifecycle
                                :style-class "reveal-form-hint"
                                :pseudo-classes #{:error}
                                :text (str/replace (impl/error result) "\n" ", ")}]
                              [])}]}}))

(defmethod event/handle ::init-popup-state [{:keys [id]}]
  #(assoc % id {:state nil
                :on-state-changed {::event/type ::change-popup-state :id id}}))

(defmethod event/handle ::dispose-popup-state [{:keys [id]}]
  #(dissoc % id))

(defmethod event/handle ::change-popup-state [{:keys [id fn]}]
  #(cond-> % (contains? % id) (update-in [id :state] fn)))

(defn- init-popup-state! [id _ handler]
  (handler {::event/type ::init-popup-state :id id})
  #(handler {::event/type ::dispose-popup-state :id id}))

(defn ^{:arglists '([{:keys [edit on-edit form desc main]
                      :or {main false}}])} view [props]
  {:fx/type rfx/ext-with-process
   :start init-popup-state!
   :desc (assoc props :fx/type popup-wrapper-impl)})

