(ns vlaaad.reveal.pro.form.json-schema
  (:require [clojure.data.json :as json]
            [clojure.java.io :as io]
            [vlaaad.reveal.pro.form.text :as text]
            [vlaaad.reveal.pro.form.entity :as entity]
            [clojure.string :as str]
            [vlaaad.reveal.pro.form.impl :as impl]
            [vlaaad.reveal.pro.form.enum :as enum]
            [vlaaad.reveal.pro.form.regex :as regex]
            [vlaaad.reveal.pro.form.delay :as delay]
            [vlaaad.reveal.pro.form.coll-of :as coll-of]
            [vlaaad.reveal.pro.form.alt :as alt]
            [vlaaad.reveal.pro.form.lazy :as lazy]))

(defn- all [explains]
  (fn [x]
    (reduce
      (fn [_ explain]
        (when-let [err (impl/explain explain x)]
          (reduced err)))
      nil
      explains)))

(defn- list-items [items last-separator]
  (->> (concat
         (->> (concat (repeat (dec (dec (count items))) ", ")
                      [(str " " last-separator " ")])
              (take (dec (count items))))
         [""])
       (interleave items)
       (apply str)))

(defn any [explains]
  (fn [x]
    (let [errs (transduce
                 (comp
                   (map #(impl/explain % x))
                   (halt-when nil?))
                 conj
                 explains)]
      (when errs
        (str "either " (list-items errs "or"))))))

(defn- one [explains]
  (fn [x]
    (let [ret (reduce
                (fn [acc explain]
                  (let [ret (if-let [err (impl/explain explain x)]
                              (update acc :errors conj err)
                              (update acc :successes inc))]
                    (if (= 2 (:successes ret))
                      (reduced ret)
                      ret)))
                {:errors []
                 :successes 0}
                explains)]
      (case (:successes ret)
        0 (str/join ", " (:errors ret))
        1 nil
        2 "too many matches"))))

(defn- explainer
  ([pred]
   (explainer pred "invalid"))
  ([pred msg]
   #(when-not (pred %) msg)))

(defn- guarded
  ([guard pred]
   (guarded guard pred "invalid"))
  ([guard pred msg]
   #(when (and (guard %)
               (not (pred %)))
      msg)))

(defn- numeric [items singular plural]
  (if (= 1 (count items))
    singular
    plural))

(defn- contains-any? [m & ks]
  (reduce
    (fn [_ k]
      (if (contains? m k)
        (reduced true)
        false))
    false
    ks))

(defn- lazy-explainer [explainer-fn]
  (let [explainer-delay (delay (explainer-fn))]
    (fn [x]
      (@explainer-delay x))))

(def ^:dynamic *root*)

(defn- pre-process [raw-schema]
  (let [{:keys [type const $ref]
         :or {const ::undefined}} raw-schema]
    (cond
      $ref
      (let [path (if (str/starts-with? $ref "#")
                   (map keyword (remove empty? (str/split (subs $ref 1) #"/")))
                   (throw (ex-info (str "Unsupported ref: " $ref " should start with #") {:$ref $ref})))]
        (pre-process (-> *root*
                         (get-in path)
                         (assoc ::ref $ref)
                         (update :title #(or % (name (or (last path) ::root)))))))

      ;; maybe separate keys from raw schema by type to specific types?
      (coll? type)
      (cond->
        {:anyOf (mapv #(-> raw-schema (assoc :type %) (dissoc :title)) type)}
        (:title raw-schema)
        (assoc :title (:title raw-schema)))

      :else
      raw-schema)))

(defn- make-explainer [{:keys [enum type allOf anyOf oneOf
                               ;; numbers and integers
                               multipleOf minimum exclusiveMinimum maximum exclusiveMaximum
                               ;; strings
                               minLength maxLength pattern
                               ;; objects
                               properties required additionalProperties
                               ;; todo validate these too
                               patternProperties dependentRequired dependentSchemas
                               minProperties maxProperties propertyNames
                               ;; array
                               prefixItems items
                               ;; todo validate these
                               contains maxContains minContains
                               maxItems minItems uniqueItems]
                        ::keys [ref]
                        not-schema :not
                        ;; todo validate these
                        if-schema :if
                        then-schema :then
                        else-schema :else
                        :as schema}]
  (let [root *root*]
    (if ref
      (lazy-explainer #(binding [*root* root]
                         (make-explainer (dissoc schema ::ref))))
      (all (cond-> []
             allOf (into (map #(-> % pre-process make-explainer)) allOf)
             anyOf (conj (any (mapv #(-> % pre-process make-explainer) anyOf)))
             oneOf (conj (one (mapv #(-> % pre-process make-explainer) oneOf)))
             not-schema (conj (let [explain (-> not-schema pre-process make-explainer)]
                                (fn [x]
                                  (if (impl/explain explain x)
                                    nil
                                    "invalid"))))
             enum (conj (let [s (set enum)]
                          (explainer #(contains? s %))))
             type (conj (case type
                          "integer" (explainer int? "should be an integer")
                          "number" (explainer number? "should be a number")
                          "string" (explainer string? "should be a string")
                          "object" (explainer map? "should be an object")
                          "boolean" (explainer boolean? "should be a boolean")
                          "null" (explainer nil? "should be null")
                          "array" (explainer vector? "should be an array")))
             ;; numbers
             multipleOf (conj (guarded number? #(zero? (mod % multipleOf))
                                       (str "should be a multiple of " multipleOf)))
             minimum (conj (guarded number? #(<= minimum %)
                                    (str "should be more or equal to " minimum)))
             exclusiveMinimum (conj (guarded number? #(< exclusiveMinimum %)
                                             (str "should be more than " exclusiveMinimum)))
             maximum (conj (guarded number? #(>= maximum %)
                                    (str "should be less or equal to " maximum)))
             exclusiveMaximum (conj (guarded number? #(> exclusiveMaximum %)
                                             (str "should be less than " exclusiveMaximum)))
             ;; strings
             minLength (conj (guarded string? #(<= minLength (count %))
                                      (str "should be at least " minLength " chars long")))
             maxLength (conj (guarded string? #(<= (count %) maxLength)
                                      (str "should be at most " maxLength " chars long")))
             pattern (conj (let [re (re-pattern pattern)]
                             (guarded string? #(re-matches re %)
                                      (str "should match " pattern))))
             ;; objects
             required (conj (let [ks (map keyword required)]
                              (fn [x]
                                (let [missing (when (map? x)
                                                (remove #(contains? x %) ks))]
                                  (when (seq missing)
                                    (str (list-items missing "and")
                                         " " (numeric missing "is" "are")
                                         " required"))))))
             minProperties (conj (guarded map? #(<= minProperties (count %))
                                          (str "should have at least " minProperties " properties")))
             maxProperties (conj (guarded map? #(>= maxProperties (count %))
                                          (str "should have at most " maxProperties " properties")))
             (false? additionalProperties) (conj (let [known-keys (set (concat
                                                                         (map keyword required)
                                                                         (keys properties)))]
                                                   ;; todo: support pattern properties
                                                   (fn [x]
                                                     (when (map? x)
                                                       (let [extras (remove known-keys (keys x))]
                                                         (when (seq extras)
                                                           (str "should not have " (list-items extras "and")
                                                                " " (numeric extras "key" "keys"))))))))
             properties (conj (let [k->explain (->> properties
                                                    (map (juxt key #(-> %
                                                                        val
                                                                        pre-process
                                                                        make-explainer)))
                                                    (into {}))]
                                (fn [x]
                                  (when (map? x)
                                    (let [invalid (keep (fn [[k explain]]
                                                          (when (contains? x k)
                                                            (let [v (get x k)]
                                                              (when-let [err (impl/explain explain v)]
                                                                [k err]))))
                                                        k->explain)]
                                      (when (seq invalid)
                                        (->> invalid
                                             (map (fn [[k err]]
                                                    (str (name k) ": " err)))
                                             (str/join ", "))))))))
             propertyNames (conj (let [explain (make-explainer (pre-process (assoc propertyNames :type "string")))]
                                   (fn [x]
                                     (when (map? x)
                                       (let [invalid (->> x
                                                          keys
                                                          (keep #(when-let [err (impl/explain explain (name %))]
                                                                   [% err])))]
                                         (when (seq invalid)
                                           (->> invalid
                                                (map (fn [[k err]]
                                                       (str k ": " err)))
                                                (str/join ", "))))))))
             ;; array
             prefixItems (conj (let [explains (mapv #(-> % pre-process make-explainer) prefixItems)]
                                 (fn [x]
                                   (when (vector? x)

                                     (let [invalid (->> x
                                                        (take (count explains))
                                                        (keep-indexed
                                                          (fn [i item]
                                                            (when-let [err (impl/explain (explains i) item)]
                                                              [i err]))))
                                           has-invalid (seq invalid)
                                           too-short (< (count x) (count explains))]
                                       (cond
                                         has-invalid (->> invalid
                                                          (map (fn [[i err]]
                                                                 (str "at " i ": " err)))
                                                          (str/join ","))
                                         too-short "should have more items"))))))

             (false? items) (conj (let [max-items (count prefixItems)]
                                    (guarded vector? #(<= (count %) max-items) "should have fewer items")))
             items (conj (let [explain (-> items pre-process make-explainer)
                               prefix-n (count prefixItems)]
                           (fn [x]
                             (when (vector? x)
                               (let [invalid (->> x
                                                  (drop prefix-n)
                                                  (keep-indexed
                                                    (fn [i item]
                                                      (when-let [err (impl/explain explain item)]
                                                        [(+ i prefix-n) err]))))]
                                 (when (seq invalid)
                                   (->> invalid
                                        (map
                                          (fn [[i err]]
                                            (str "at " i ": " err)))
                                        (str/join ", "))))))))
             minItems (conj (guarded vector? #(<= minItems (count %))
                                     (str "should have at least " minItems " items")))
             maxItems (conj (guarded vector? #(>= maxItems (count %))
                                     (str "should have at most " maxItems " items")))
             uniqueItems (conj (guarded vector? #(apply distinct? %)
                                        (str "items must be unique"))))))))

(defn- infer-label [{:keys [title type anyOf oneOf allOf]
                     not-schema :not-schema
                     :as schema}]
  (or title
      (when (:enum schema)
        "enum")
      (when (contains-any? schema :multipleOf :minimum :exclusiveMinimum :maximum :exclusiveMaximum)
        "number")
      (when (contains-any? schema :minLength :maxLength :pattern)
        "string")
      (when (contains-any? schema :prefixItems :items :contains :maxContains :minContains :maxItems
                 :minItems :uniqueItems)
        (let [{:keys [items]} schema]
          (str "array"
               (when items
                 (str " of " (infer-label (pre-process items)))))))
      type
      (when (contains-any? schema :properties :required :additionalProperties :patternProperties
                           :dependentRequired :dependentSchemas :minProperties :maxProperties
                           :propertyNames)
        "object")
      (when oneOf
        (str/join " or " (distinct (map #(-> % pre-process infer-label) oneOf))))
      (when anyOf
        (str/join " or " (distinct (map #(-> % pre-process infer-label) anyOf))))
      (when allOf
        (str/join " and " (distinct (map #(-> % pre-process infer-label) allOf))))
      (when not-schema
        (str "not " (infer-label (pre-process not-schema))))
      "value"))

(defn- entity? [schema]
  (contains-any? schema :required :properties :additionalProperties))

(defn- merge-entities [schemas]
  (reduce
    (fn [acc schema]
      (-> acc
          (update :required into (:required schema))
          (update :properties #(merge-with merge %1 %2) (:properties schema))
          (update :additionalProperties #(and % (:additionalProperties schema true)))))
    {::editor :entity
     :required []
     :properties {}
     :additionalProperties true}
    schemas))

(defn- relax-entity [schema]
  (-> schema
      (dissoc :required)
      (update :properties
              #(merge-with merge %1 %2)
              (zipmap
                (->> schema :required (map keyword))
                (repeat {})))))

(defn- select-all [all]
  (let [entities (into [] (filter #(-> % ::editor (= :entity))) all)]
    (or (when (pos? (count entities))
          (merge-entities entities))
        (some #(when-not (-> % ::editor (= :text)) %) all)
        (first all))))

(defn select-editor [schema]
  ;; TODO: bring props from if/then/else/dependentSchemas
  ;; https://json-schema.org/understanding-json-schema/reference/conditionals.html
  (cond
    (:enum schema)
    (assoc schema ::editor :enum)

    (= "null" (:type schema))
    (assoc schema ::editor :enum :enum [nil])

    (contains? schema :const)
    (assoc schema ::editor :enum :enum [(:const schema)])

    (or (:oneOf schema) (:anyOf schema))
    (let [any (mapv #(-> % pre-process select-editor)
                    (concat (:oneOf schema) (:anyOf schema)))]
      (select-all
        [(select-editor (dissoc schema :oneOf :anyOf))
         (if (every? #(-> % ::editor (= :entity)) any)
           (merge-entities (map relax-entity any))
           (assoc schema ::editor :alt :alts any))]))

    (:allOf schema)
    (select-all
      ;; TODO: should use something other than select-editor here? we know it's not an enum...
      (into [(select-editor (dissoc schema :allOf))]
            (map #(-> % pre-process select-editor))
            (:allOf schema)))

    (entity? schema)
    (assoc schema ::editor :entity)

    (:prefixItems schema)
    (assoc schema ::editor :regex)

    (:items schema)
    (assoc schema ::editor :coll)

    :else
    (assoc schema ::editor :text)))

(defn- load-json-schema [json]
  (if (map? json)
    json
    (with-open [rdr (io/reader json)]
      (json/read rdr :key-fn keyword))))

(defn make-form [json]
  (letfn [(->form-editor [schema visited']
            (let [ed (select-editor schema)]
              (case (::editor ed)
                :enum
                (enum/value-editor {:values (:enum ed)})

                :alt
                (let [{:keys [alts]} ed]
                  (alt/value-editor {:alts (mapv #((juxt :label identity) (->form % visited')) alts)}))

                :entity
                (let [{:keys [required properties additionalProperties]} ed]
                  (let [forms (->> properties
                                   (map (juxt key #(-> % val (->form visited'))))
                                   (into {}))
                        any-form (->form {} visited')
                        req-keys (map keyword required)]
                    (entity/value-editor
                      {:req (->> req-keys
                                 (map (juxt identity #(get forms % any-form)))
                                 (into {}))
                       :opt (sort-by #(-> % key str) (apply dissoc forms req-keys))
                       :any (when additionalProperties
                              (constantly any-form))})))

                :regex
                (let [{:keys [prefixItems items minItems maxItems]} ed
                      prefix-n (count prefixItems)
                      min-items (-> minItems (or 0) (- prefix-n) (max 0))
                      max-items (-> maxItems (or ##Inf) (- prefix-n) (max min-items))]
                  (regex/value-editor
                    {:kinds [:vector]
                     :op (regex/cat {:cat-ops (vec
                                                (concat
                                                  (->> prefixItems
                                                       (map #(->form % visited'))
                                                       (map-indexed vector))
                                                  (when items
                                                    (let [op (cond
                                                               (and (zero? min-items)
                                                                    (Double/isInfinite max-items))
                                                               regex/*
                                                               (and (zero? min-items)
                                                                    (= 1 max-items))
                                                               regex/?
                                                               (and (= 1 min-items)
                                                                    (Double/isInfinite max-items))
                                                               regex/+
                                                               :else
                                                               #(regex/repeat (assoc % :min-count min-items
                                                                                       :max-count max-items)))]
                                                      [[:items (op {:op (->form items visited')
                                                                    :parse #(mapv vector %)})]]))))
                                     :parse (let [init (zipmap
                                                         (range prefix-n)
                                                         (repeat []))]
                                              (fn [xs]
                                                (let [ret (->> xs
                                                               (take prefix-n)
                                                               (map vector)
                                                               (map-indexed vector)
                                                               (into init))]
                                                  (cond-> ret
                                                    items
                                                    (assoc :items (vec (drop prefix-n xs)))))))})}))

                :coll
                (let [{:keys [items minItems maxItems]} ed]
                  (coll-of/value-editor
                    (cond-> {:kinds [:vector]
                             :item-form (->form items visited')}
                      minItems (assoc :min-count minItems)
                      maxItems (assoc :max-count maxItems))))

                :text
                text/value-editor)))
          (->form [raw-schema visited]
            (let [{:keys [description default examples]
                   ::keys [ref]
                   :as schema} (pre-process raw-schema)
                  visited' (cond-> visited ref (conj ref))
                  root *root*]
              (cond->
                {:label (infer-label schema)
                 :explain (make-explainer schema)
                 :editor (if (visited ref)
                           (lazy/value-editor #(binding [*root* root] (->form-editor schema #{})))
                           (delay/value-editor
                             #(binding [*root* root] (->form-editor schema visited'))))
                 :options (cond-> []
                            (contains? schema :default)
                            (conj {:label "Default value"
                                   :invoke (constantly default)
                                   :shortcut [:shortcut :d]})
                            examples
                            (into (comp
                                    (take 9)
                                    (map-indexed
                                      (fn [i example]
                                        (let [n (inc i)]
                                          {:label (str "Example " n)
                                           :invoke (constantly example)
                                           :shortcut [:shortcut (keyword (str "DIGIT" n))]}))))
                                  examples))}
                description
                (assoc :description description))))]
    (binding [*root* (load-json-schema json)]
      (->form *root* #{}))))
