(ns vlaaad.reveal.pro.form.impl
  "© 2021 Vladislav Protsenko. All rights reserved."
  (:require [clojure.main :as m]
            [clojure.pprint :as pp]
            [clojure.string :as str]
            [vlaaad.reveal.font :as font]
            [cljfx.api :as fx]
            [cljfx.prop :as fx.prop]
            [cljfx.mutator :as fx.mutator]
            [cljfx.lifecycle :as fx.lifecycle])
  (:import [javafx.util StringConverter]
           [javafx.scene.input KeyEvent KeyCode]
           [javafx.scene Node Parent]
           [javafx.scene.control Skinnable]
           [javafx.beans.value ChangeListener]))

(def undefined '???)

(defn completely-undefined?
  "No value at all"
  [x]
  (= undefined x))

(defn partially-undefined?
  "There might be a value, but it's not enough to do validation"
  [x]
  (cond
    (completely-undefined? x) true
    (nil? x) false
    (string? x) false
    (seqable? x) (boolean (some partially-undefined? x))
    :else false))

(defn explain
  "Validates possibly undefined value and returns either nil (valid) or error string

  explain is a function that given a defined value, either:
  - returns string error message or throws exception if invalid
  - returns nil if valid"
  [explain value]
  (if (partially-undefined? value)
    "required"
    (try
      (explain value)
      (catch Exception e
        (-> e Throwable->map m/ex-triage (:clojure.error/cause "invalid"))))))

(defn valid? [form x]
  (nil? (explain (:explain form) x)))

(defprotocol Editor
  :extend-via-metadata true
  (^:private edit* [editor value]
    "Returns Edit of (possibly undefined) value"))

(defn edit
  ([editor]
   (edit* editor undefined))
  ([editor x]
   (edit* editor x)))

(defprotocol Result
  :extend-via-metadata true
  (value [result] "Returns possibly undefined value corresponding to a result")
  (error [result] "Returns error string or nil if valid"))

(extend-protocol Result
  nil
  (value [_] nil)
  (error [_] nil)
  Object
  (value [o] o)
  (error [o] nil))

(defn error-result
  ([error]
   (error-result undefined error))
  ([value error]
   (with-meta {:value value :error error} {`value :value `error :error})))

(defprotocol Edit
  :extend-via-metadata true
  (^:private assemble* [edit]
    "Attempt to convert edit back to data, returns Result or throws")
  (^:private view [edit]
    "Returns cljfx view fn that will receive edit and on-edit"))

(defn assemble
  "Attempt to convert edit back to value, without any validation. Returns Result

  Partially undefined values are not errors"
  [edit]
  (try
    (assemble* edit)
    (catch Exception e
      (error-result (-> e Throwable->map m/ex-triage (:clojure.error/cause "unreadable"))))))

(defn validate [explain-fn edit]
  (let [result (assemble edit)]
    (if (error result)
      result
      (if-let [error (explain explain-fn (value result))]
        (error-result (value result) error)
        result))))

(defn form-view [{:keys [form edit on-edit]}]
  {:fx/type (view edit)
   :edit edit
   :on-edit on-edit
   :form form})

(defn content-view [{:keys [edit on-edit]}]
  {:fx/type (view edit)
   :edit edit
   :on-edit on-edit})

(defn- on-menu-button-key-pressed [^KeyEvent e]
  (when (and (.isConsumed e)
             (= KeyCode/ESCAPE (.getCode e)))
    (.fireEvent (.getParent ^javafx.scene.Node (.getTarget e))
                (KeyEvent.
                  (.getSource e)
                  (.getTarget e)
                  (.getEventType e)
                  (.getCharacter e)
                  (.getText e)
                  (.getCode e)
                  (.isShiftDown e)
                  (.isControlDown e)
                  (.isAltDown e)
                  (.isMetaDown e)))))

(defn menu-button [props]
  (assoc props
    :fx/type :menu-button
    :on-key-pressed on-menu-button-key-pressed
    :style-class "reveal-form-input"
    :content-display :right
    :graphic-text-gap (font/char-width)
    :graphic {:fx/type :region
              :style-class "reveal-form-arrow-down"}))

(defn- edit-map-editor [{:keys [edit assemble view]} value]
  (with-meta (edit value) {`assemble* assemble `view (fn [_] view)}))

(defn ^{:arglists '([& {:keys [edit assemble view]}])} make-editor
  "Keys:
  - :edit is a function that transforms (partially defined) value to edit base (IObj)
  - :assemble is a function that transforms edit back to value
  - :view is a cljfx component that will receive :edit, :on-edit (and :form if value
    editor) props"
  [& {:as opts}]
  (with-meta opts {`edit* edit-map-editor}))

(defn pprint-str [x]
  (str/trim-newline (with-out-str (pp/pprint x))))

(defn indent-view [{:keys [desc]}]
  {:fx/type :h-box
   :children [{:fx/type :region
               :min-width (* 2 (font/char-width))}
              desc]})

(defn ^{:arglists '([{:keys [children]}])} multi-line-view [props]
  (assoc props :fx/type :v-box
               :spacing 1))

(defn assemble-all
  "Assemble multiple edits (map with edit vals or vector of edits) to a single value"
  [edits]
  (let [{:keys [errors values]} (reduce-kv
                                  (fn [acc k item-edit]
                                    (let [result (assemble item-edit)
                                          error (error result)]
                                      (-> acc
                                          (update :values assoc k (value result))
                                          (cond-> error (update :errors conj [k error])))))
                                  {:errors []
                                   :values (empty edits)}
                                  edits)]
    (if (pos? (count errors))
      (error-result
        values
        (->> errors
             (map (fn [[i err]]
                    (str "at " i ": " err)))
             (str/join ", ")))
      values)))

(defn fmap-result [result f]
  (let [val (value result)
        err (error result)
        [new-val new-err] (try
                            [(f val) nil]
                            (catch Exception e
                              [undefined (-> e
                                             Throwable->map
                                             m/ex-triage
                                             (:clojure.error/cause "unreadable"))]))]
    (if (or err new-err)
      (error-result new-val (str/join ", " (remove nil? [new-err err])))
      new-val)))

(defn- descendant-seq [^Node node]
  (cons node (when (instance? Parent node)
               (mapcat descendant-seq (.getChildrenUnmodifiable ^Parent node)))))

(defn- focus! [^Node node]
  (fx/run-later
    (when (instance? Skinnable node)
      (.applyCss node))
    (let [^Node node (or (->> node
                              descendant-seq
                              (some #(when (.isFocusTraversable ^Node %) %)))
                         node)]
      (.requestFocus node))))

(defn- focus-when-on-scene! [^Node node]
  (if (some? (.getScene node))
    (focus! node)
    (.addListener (.sceneProperty node)
                  (reify ChangeListener
                    (changed [this _ _ new-scene]
                      (when (some? new-scene)
                        (.removeListener (.sceneProperty node) this)
                        (focus! node)))))))

(def ^:private ext-with-focused
  (fx/make-ext-with-props
    {:focused (fx.prop/make
                (fx.mutator/setter
                  (fn [^javafx.scene.Node node focused]
                    (when focused
                      (focus-when-on-scene! node))))
                fx.lifecycle/scalar)}))

(defn ext-with-focus-on-request [{:keys [desc focused]}]
  {:fx/type ext-with-focused
   :props {:focused focused}
   :desc desc})