(ns vlaaad.reveal.pro.form
  "© 2021 Vladislav Protsenko. All rights reserved."
  (:refer-clojure :exclude [* + cat repeat])
  (:require [vlaaad.reveal.pro.form.impl :as impl]
            [vlaaad.reveal.pro.form.state :as state]
            [vlaaad.reveal.pro.form.alt :as alt]
            [vlaaad.reveal.pro.form.tuple :as tuple]
            [vlaaad.reveal.pro.form.text :as text]
            [vlaaad.reveal.pro.form.enum :as enum]
            [vlaaad.reveal.pro.form.coll-of :as coll-of]
            [vlaaad.reveal.pro.form.spec-alpha :as spec-alpha]
            [vlaaad.reveal.pro.form.map-of :as map-of]
            [vlaaad.reveal.pro.form.entity :as entity]
            [vlaaad.reveal.pro.form.regex :as regex]
            [vlaaad.reveal.pro.form.specs :as specs]
            [vlaaad.reveal.action :as action]
            [vlaaad.reveal.view :as view]
            [clojure.spec.alpha :as s]))

;; region form

(defn form
  "Create a form - validator and editor for a value

  This is a convenience function since form is just a map with these keys:
  - :label (required) - short string description of edited value
  - :explain (required) - value validator, a 1-arg function that should either:
    - return nil if value is valid
    - return string description of an error if value is invalid
    - throw exception of value is invalid
  - :editor (required) - an editor instance used to edit the value
  - :description (optional) - longer string description of edited value
  - :options (optional) - a vector of options available in context menu,
    represented as maps expecting these keys:
    - :label (required) - option label
    - :invoke (required) - 0-arg function that produces a value to edit
    - :shortcut (optional) - cljfx key combination description"
  [& {:keys [explain editor label description options] :as opts}]
  (specs/validate! ::specs/form opts)
  opts)

;; endregion

;; region state

(defn state
  "Create a ref with form state that is edited using [[state-view]]

  A value in a ref is either:
  - a map with a single :error key and error string value
  - a map with a single :value key and valid form value"
  ([form]
   (specs/validate! ::specs/form form)
   (state form impl/undefined))
  ([form value]
   (specs/validate! ::specs/form form)
   (state/state form value)))

;; endregion

;; region value editors

(def text-editor
  "Text field editor that expects user to enter a string to [[read]]"
  text/value-editor)

(defn alt-editor
  "Create drop down editor that allows user to use different forms for editing

  Expected args:
  - :alts (required) - key-value pairs (e.g. a map or a sequence of tuples)
    where keys are anything and values are [[form]] instances"
  [& {:keys [alts] :as opts}]
  (specs/validate! ::specs/alt-editor-args opts)
  (alt/value-editor opts))

(defn tuple-editor
  "Create editor allowing editing a tuple - vector of heterogeneous items

  Expected args:
  - :forms - a sequential coll of [[form]] items"
  [& {:keys [forms] :as opts}]
  (specs/validate! ::specs/tuple-editor-args opts)
  (tuple/value-editor opts))

(defn enum-editor
  "Create a drop down editor allowing selecting a value from a set of choices

  Expected args:
  - :values - a coll of possible values"
  [& {:keys [values] :as opts}]
  (specs/validate! ::specs/enum-editor-args opts)
  (enum/value-editor opts))

(defn coll-editor
  "Create editor allowing editing a coll of homogeneous items

  Expected args:
  - :item-form (required) - a [[form]] for individual items
  - :kinds (optional) - possible coll types, a distinct coll of either
    :vector, :list, :map or :set
  - :min-count (optional) - minimum number of items in a coll
  - :max-count (optional) - maximum number of items in a coll, inclusive"
  [& {:keys [item-form kinds min-count max-count] :as opts}]
  (specs/validate! ::specs/coll-editor-args opts)
  (coll-of/value-editor opts))

(defn map-editor
  "Create editor allowing editing a map of homogeneous keys and vals

  Expected args:
  - :key-form (required) - a [[form]] for keys
  - :val-form (required) - a [[form]] for vals
  - :min-count (optional) - minimum number of entries in a map
  - :max-count (optional) - maximum number of entries in a map, inclusive"
  [& {:keys [key-form val-form min-count max-count] :as opts}]
  (specs/validate! ::specs/map-editor-args opts)
  (map-of/value-editor opts))

(defn entity-editor
  "Create editor allowing editing an entity map with heterogeneous keys and vals

  Expected args:
  - :req (optional) - required keys in a map, key-value pairs (e.g. a map
    or a sequence of tuples) where keys are anything and values are [[form]]
    instances
  - :opt (optional) - optional keys in a map, key-value pairs like :req
  - :any (optional) - a function of a key that is not present in either :req
    or :opt that returns a form for that key"
  [& {:keys [req opt any] :as opts}]
  (specs/validate! ::specs/entity-editor-args opts)
  (entity/value-editor opts))

(defn regex-editor
  "Create editor for a sequential coll with items encoded as a data regex

  Expected args:
  - :op (required) - regex op, either a [[form]] (describing a single item in
    a coll) or one of [[*]], [[?]], [[+]], [[repeat]], [[entity]], [[alt]] and
    [[cat]]
  - :kinds (optional) - possible sequential coll types, a distinct coll
    of either :vector or :list"
  [& {:keys [kinds op] :as opts}]
  (specs/validate! ::specs/regex-editor-args opts)
  (regex/value-editor opts))

;; endregion

;; region regex ops

(defn *
  "Create regex op for a 0 or more repetitions of input op

  Expected args:
  - :op (required) - regex op, either a [[form]] (describing a single item in
    a coll) or one of [[*]], [[?]], [[+]], [[repeat]], [[entity]], [[alt]] and
    [[cat]]
  - :parse (required) - a function that splits a sequential coll of items to a
    sequential coll of colls of items, where each coll of items is edited by
    the provided op"
  [& {:keys [op parse] :as opts}]
  (specs/validate! ::specs/*-args opts)
  (regex/* opts))

(defn ?
  "Create regex op for a 0 or 1 repetitions of input op

  Expected args:
  - :op (required) - regex op, either a [[form]] (describing a single item in
    a coll) or one of [[*]], [[?]], [[+]], [[repeat]], [[entity]], [[alt]] and
    [[cat]]
  - :parse (required) - a function that splits a sequential coll of items to a
    sequential coll of colls of items, where each coll of items is edited by
    the provided op"
  [& {:keys [op parse] :as opts}]
  (specs/validate! ::specs/?-args opts)
  (regex/? opts))

(defn +
  "Create regex op for a 1 or more repetitions of input op

  Expected args:
  - :op (required) - regex op, either a [[form]] (describing a single item in
    a coll) or one of [[*]], [[?]], [[+]], [[repeat]], [[entity]], [[alt]] and
    [[cat]]
  - :parse (required) - a function that splits a sequential coll of items to a
    sequential coll of colls of items, where each coll of items is edited by
    the provided op"
  [& {:keys [op parse] :as opts}]
  (specs/validate! ::specs/+-args opts)
  (regex/+ opts))

(defn repeat
  "Create regex op for a upper and/or lower bounded repetition of input op

  Expected args:
  - :op (required) - regex op, either a [[form]] (describing a single item in
    a coll) or one of [[*]], [[?]], [[+]], [[repeat]], [[entity]], [[alt]] and
    [[cat]]
  - :parse (required) - a function that splits a sequential coll of items to a
    sequential coll of colls of items, where each coll of items is edited by
    the provided op
  - :min-count (optional) - minimum number of op repetitions
  - :max-count (optional) - maximum number of op repetitions, inclusive"
  [& {:keys [op parse min-count max-count] :as opts}]
  (specs/validate! ::specs/repeat-args opts)
  (regex/repeat opts))

(defn entity
  "Create regex op for a sequentially encoded heterogenenous key-value pairs

  Expected args:
  - :req (optional) - required keys in a kv seq, key-value pairs (e.g. a map
    or a sequence of tuples) where keys are anything and values are [[form]]
    instances
  - :opt (optional) - optional keys in a kv seq, key-value pairs like :req
  - :any (optional) - a function of a key that is not present in either :req
    or :opt that returns a form for that key"
  [& {:keys [req opt any] :as opts}]
  (specs/validate! ::specs/entity-args opts)
  (regex/entity opts))

(defn alt
  "Create regex op that allows selecting different op paths for editing

  Expected args:
  - :alt-ops (required) - key-value pairs (e.g. a map or a sequence of tuples)
    where keys are anything and values are regex ops, i.e. either a [[form]]
    (describing a single item in a coll) or one of [[*]], [[?]], [[+]],
    [[repeat]], [[entity]], [[alt]] and [[cat]]
  - :parse (required) - a function that splits a sequential coll of items to
    a tuple of alt key from :alt-ops and that same coll of items"
  [& {:keys [alt-ops parse] :as opts}]
  (specs/validate! ::specs/alt-args opts)
  (regex/alt opts))

(defn cat
  "Create regex op for a sequential concatenation of different regex ops

  Expected args:
  - :cat-ops (required) - sequential key-value pairs where keys are anything
    and values are regex ops, i.e. either a [[form]] (describing a single item
    in a coll) or one of [[*]], [[?]], [[+]], [[repeat]], [[entity]], [[alt]]
    and [[cat]]
  - :parse (required) - a function that splits sequential coll of items to a map
    with :cat-ops keys as keys and sequences of items for :cat-ops vals as vals"
  [& {:keys [cat-ops parse] :as opts}]
  (specs/validate! ::specs/cat-args opts)
  (regex/cat opts))

;; endregion

;; region spec-alpha

(defn spec-alpha-form
  "Convert clojure.spec.alpha spec to a [[form]]"
  [spec]
  (spec-alpha/make-form spec))

;; endregion

;; region simple views

(defn state-view
  "Cljfx component fn that shows UI for editing a [[state]] defined by its [[form]]

  Expected props:
  - :state (required) - a [[state]] ref"
  [{:keys [state]}]
  {:fx/type state/view
   :state state})

;; endregion

;; region easy views

(defn form-view
  "Cljfx component fn that shows UI for editing a [[form]]

  This is a convenience component that manages the state using cljfx. This means
  that any change to props of this component will clear the form

  Expected props:
  - :form (required) - a [[form]]
  - :value (optional) - initial value edited by the form"
  [{:keys [form value]
    :or {value impl/undefined}}]
  {:fx/type state-view
   :state (state form value)})

(defn observable-value-view
  "Cljfx component fn that allows to define a view derived from a form value

  This is a convenience component that shows text error as long as form is
  invalid and a cljfx component produced by provided fn when the form is valid

  Expected props:
  - :state (required) - a [[state]] ref
  - :fn (required) - a function of a value successfully validated by state's
    form that returns a cljfx component"
  [{:keys [state fn]}]
  {:fx/type view/observable-view
   :ref state
   :fn (clojure.core/fn [{:keys [error value]}]
         (if error
           {:fx/type :scroll-pane
            :content {:fx/type :label
                      :style-class "reveal-form-error"
                      :wrap-text true
                      :text error}}
           (fn value)))})

(defn split-form-view
  "Cljfx component fn that shows a split view with a form UI and another view
  derived from a form value

  This is a convenience component that can be easily assembled from other views
  like [[state-view]] and [[observable-value-view]]

  Expected props:
  - :form (required) - a [[form]]
  - :fn (required) - a function of a value successfully validated by state's
    form that returns a cljfx component
  - :value (optional) - initial value edited by the form
  - :orientation (optional, default :vertical) - a split orientation, either
    :vertical or :horizontal"
  [{:keys [form fn value orientation]
    :or {value impl/undefined
         orientation :vertical}}]
  (let [s (state form value)]
    {:fx/type :split-pane
     :divider-positions [0.5]
     :orientation orientation
     :items [{:fx/type state-view
              :state s}
             {:fx/type observable-value-view
              :state s
              :fn fn}]}))

;; endregion

;; region actions

(action/defaction ::action/form:spec [x]
  (when (or (s/get-spec x)
            (s/spec? x)
            (s/regex? x))
    (fn []
      {:fx/type form-view
       :form (spec-alpha-form x)})))

;; endregion