(ns vlaaad.reveal.pro.form.regex
  "© 2021 Vladislav Protsenko. All rights reserved."
  (:refer-clojure :exclude [* + cat repeat])
  (:require [vlaaad.reveal.pro.form.impl :as impl]
            [vlaaad.reveal.event :as event]
            [vlaaad.reveal.pro.form.coll-of :as coll-of]
            [vlaaad.reveal.pro.form.text :as text]
            [vlaaad.reveal.pro.form.vignette :as vignette]
            [vlaaad.reveal.pro.form.entity :as entity]
            [vlaaad.reveal.font :as font]
            [clojure.string :as str]
            [vlaaad.reveal.pro.form.compatible :as compatible]
            [vlaaad.reveal.pro.form.copy-paste :as copy-paste]))

;; region parsing

(def ^:private unparsed-editor
  (coll-of/content-editor {:item-form {:label "???"
                                       :explain (constantly "unparsed")
                                       :editor text/value-editor}}))

(defn- edit-parsed [pref-content-editor parse x]
  (try
    {:pref (impl/edit
             pref-content-editor
             (if (impl/completely-undefined? x)
               x
               (parse x)))}
    (catch Exception _
      {:coll (impl/edit unparsed-editor x)})))

(defn- assemble-parsed [unparse {:keys [pref coll]}]
  (if pref
    (impl/fmap-result (impl/assemble pref) unparse)
    (impl/assemble coll)))

(defmethod event/handle ::edit-pref-edit [{:keys [on-edit fn]}]
  (event/handle (assoc on-edit :fn #(update % :pref fn))))

(defmethod event/handle ::edit-coll-edit [{:keys [on-edit pref-content-editor parse fn]}]
  (event/handle
    (assoc on-edit :fn #(let [coll-edit (fn (:coll %))
                              result (impl/assemble coll-edit)
                              v (impl/value result)]
                          (if (impl/error result)
                            (assoc % :coll coll-edit)
                            (try
                              (-> %
                                  (dissoc :coll)
                                  (assoc :pref (impl/edit
                                                 pref-content-editor
                                                 (if (impl/completely-undefined? v)
                                                   v
                                                   (parse v)))))
                              (catch Exception _
                                (assoc % :coll coll-edit))))))))

(defn- view-parsed [pref-content-editor parse {:keys [edit on-edit]}]
  (let [{:keys [pref coll]} edit]
    (if pref
      {:fx/type impl/content-view
       :edit pref
       :on-edit {::event/type ::edit-pref-edit
                 :on-edit on-edit}}
      {:fx/type impl/content-view
       :edit coll
       :on-edit {::event/type ::edit-coll-edit
                 :on-edit on-edit
                 :pref-content-editor pref-content-editor
                 :parse parse}})))

(defn- wrap-parse-content-editor
  "edit sequential of items internally as something else

  `editor` edits parsed something else. if unparseable, editor edits a coll of any items
  `parse` attempts to convert input to something else or throws"
  [pref-content-editor parse unparse]
  (impl/make-editor
    :edit #(edit-parsed pref-content-editor parse %)
    :assemble #(assemble-parsed unparse %)
    :view #(view-parsed pref-content-editor parse %)))

;; endregion

;; region ->form

(defmethod event/handle ::edit-content [{:keys [on-edit fn]}]
  (event/handle (assoc on-edit :fn #(update % :content fn))))

(defn- edit-slice [content-editor x]
  {:content (impl/edit content-editor x)})

(defn- assemble-slice [{:keys [content]}]
  (impl/assemble content))

(defn- view-slice [{:keys [form edit on-edit]}]
  {:fx/type impl/multi-line-view
   :children [{:fx/type vignette/view
               :edit edit
               :on-edit on-edit
               :form form
               :main true
               :desc {:fx/type :button
                      :style-class "reveal-form-input"
                      :pseudo-classes #{:control}
                      :text "["}}
              {:fx/type impl/indent-view
               :desc {:fx/type impl/content-view
                      :edit (:content edit)
                      :on-edit {::event/type ::edit-content
                                :on-edit on-edit}}}
              {:fx/type :label
               :style-class "reveal-form-label"
               :pseudo-classes #{:control}
               :text "]"}]})

(defn- content-editor-op->form [{:keys [label] :as content-editor}]
  {:label label
   :explain (constantly nil)
   :editor (impl/make-editor
             :edit #(edit-slice content-editor %)
             :assemble assemble-slice
             :view view-slice)})

(defn- edit-singleton [editor x]
  {:content (if (or (impl/completely-undefined? x)
                    (empty? x))
              (impl/edit editor)
              (impl/edit editor (first x)))})

(defn- assemble-singleton [{:keys [content]}]
  (impl/fmap-result (impl/assemble content) vector))

(defn- view-singleton [initial-form {:keys [form edit on-edit]}]
  {:fx/type impl/form-view
   :form (assoc initial-form :options (:options form []))
   :edit (:content edit)
   :on-edit {::event/type ::edit-content :on-edit on-edit}})

(defn- wrap-singleton-editor
  "wraps editor so it edits a singleton coll of item edited by editor"
  [value-editor initial-form]
  (impl/make-editor
    :edit #(edit-singleton value-editor %)
    :assemble assemble-singleton
    :view #(view-singleton initial-form %)))

(defn- wrap-singleton-explain [explain]
  (fn [x]
    (explain (first x))))

(defn- form-op->singleton-form [form]
  (-> form
      (update :explain wrap-singleton-explain)
      (update :editor wrap-singleton-editor form)))

(defn- op->item-form [op]
  (if (:explain op)
    (form-op->singleton-form op)
    (content-editor-op->form op)))

;; endregion

;; region ->content-editor

(defn- edit-form [{:keys [editor]} x]
  {:content (impl/edit editor x)})

(defn- assemble-form [{:keys [content]}]
  (impl/assemble content))

(defn- view-form [form {:keys [edit on-edit]}]
  {:fx/type impl/form-view
   :form form
   :edit (:content edit)
   :on-edit {::event/type ::edit-content :on-edit on-edit}})

(defn- form->content-editor [form]
  (impl/make-editor
    :edit #(edit-form form %)
    :assemble assemble-form
    :view #(view-form form %)))

(defn- op->content-editor [op]
  (if (:explain op)
    (form->content-editor (form-op->singleton-form op))
    op))

;; endregion

;; region repeat

(defn- repeat-impl
  "
  `:op` is either:
   - item form (T coerced to singleton [T] slice)
   - sequential content editor with :label
  `:parse` needs to parse [T] to [[T]]

  edits [T] by parsing it to [[T]] and editing individual [T]s using op"
  [{:keys [op parse min-count max-count suffix]}]
  (-> (coll-of/content-editor
        {:item-form (op->item-form op)
         :min-count min-count
         :max-count max-count})
      (wrap-parse-content-editor parse #(into [] clojure.core/cat %))
      (assoc :label (str "(" (:label op) ")" suffix))))

(defn ^{:arglists '([{:keys [op parse]}])} * [opts]
  (-> opts
      (assoc :min-count 0
             :max-count Integer/MAX_VALUE
             :suffix "*")
      (repeat-impl)))

(defn ^{:arglists '([{:keys [op parse]}])} ? [opts]
  (-> opts
      (assoc :min-count 0
             :max-count 1
             :suffix "?")
      (repeat-impl)))

(defn ^{:arglists '([{:keys [op parse]}])} + [opts]
  (-> opts
      (assoc :min-count 1
             :max-count Integer/MAX_VALUE
             :suffix "+")
      (repeat-impl)))

(defn ^{:arglists '([{:keys [op parse min-count max-count]}])} repeat
  [{:keys [min-count max-count] :as opts}]
  (-> opts
      (assoc :min-count min-count
             :max-count max-count
             :suffix (str "{"
                          min-count
                          (when-not (= min-count max-count)
                            (str "," max-count))
                          "}"))
      (repeat-impl)))

;; endregion

;; region entity

(defn- edit-entity [map-editor kvs]
  {:content (impl/edit
              map-editor
              (into {}
                    (comp
                      (partition-all 2)
                      (map #(cond-> % (= 1 (count %)) (conj impl/undefined))))
                    (if (impl/completely-undefined? kvs)
                      []
                      kvs)))})

(defn- assemble-entity [{:keys [content]}]
  (impl/fmap-result (impl/assemble content) #(into [] clojure.core/cat %)))

(defn- entity-view [{:keys [edit on-edit]}]
  {:fx/type impl/content-view
   :edit (:content edit)
   :on-edit {::event/type ::edit-content :on-edit on-edit}})

(defn ^{:arglists '([{:keys [req opt any]}])} entity [opts]
  (let [e (entity/content-editor opts)]
    (-> (impl/make-editor
          :edit #(edit-entity e %)
          :assemble assemble-entity
          :view entity-view)
        (assoc :label "kvs"))))

;; endregion

;; region alt

(defn- edit-alt [alts x]
  (if (impl/completely-undefined? x)
    (let [[k e] (first alts)]
      {:key k
       :edits {k (impl/edit e)}})
    (let [[k xs] x
          e (some #(when (= k (first %)) (second %)) alts)]
      {:key k
       :edits {k (impl/edit e xs)}})))

(defn- assemble-alt [{:keys [key edits]}]
  (impl/fmap-result (impl/assemble (get edits key))
                    #(vector key %)))

(defmethod event/handle ::select-alt [{:keys [alt on-edit]}]
  (let [[k e] alt]
    (event/handle
      (assoc on-edit :fn (fn [edit]
                           (-> edit
                               (assoc :key k)
                               (cond-> (not (contains? (:edits edit) k))
                                 (assoc-in [:edits k] (impl/edit e)))))))))

(defmethod event/handle ::edit-alt [{:keys [on-edit fn key]}]
  (event/handle (assoc on-edit :fn #(update-in % [:edits key] fn))))

(defn- view-alt [alts {:keys [edit on-edit]}]
  (let [{:keys [key edits]} edit]
    {:fx/type impl/multi-line-view
     :children [{:fx/type impl/menu-button
                 :pseudo-classes #{:control}
                 :text (pr-str key)
                 :items (for [[k e :as alt] alts]
                          {:fx/type :menu-item
                           :on-action {::event/type ::select-alt
                                       :on-edit on-edit
                                       :alt alt}
                           :graphic {:fx/type :h-box
                                     :spacing (font/char-width)
                                     :children [{:fx/type :label
                                                 :style-class "reveal-form-label"
                                                 :pseudo-classes #{:control}
                                                 :text (pr-str k)}
                                                {:fx/type :label
                                                 :style-class "reveal-form-label"
                                                 :pseudo-classes #{:hint}
                                                 :text (:label e)}]}})}
                {:fx/type impl/indent-view
                 :desc {:fx/type impl/content-view
                        :edit (get edits key)
                        :on-edit {::event/type ::edit-alt
                                  :on-edit on-edit
                                  :key key}}}]}))

(defn alt
  "`:alt-ops` are kvs from any key (alt) to op (either single value form
  or sequential content editor with :label)

  `:parse` needs to convert [T] to [alt-key [T]] (where result [T] is the same)"
  [{:keys [alt-ops parse]}]
  (let [alts (mapv
               (fn [[k op]]
                 [k (op->content-editor op)])
               alt-ops)]
    (-> (impl/make-editor
          :edit #(edit-alt alts %)
          :assemble assemble-alt
          :view #(view-alt alts %))
        (wrap-parse-content-editor parse second)
        (assoc :label (str/join "|" (->> alts (map first) (map pr-str)))))))

;; endregion

;; region cat

(defn- edit-cat [cat-editors m]
  (let [->vals (if (impl/completely-undefined? m)
                 (constantly impl/undefined)
                 m)]
    (->> cat-editors
         (map (fn [[k editor]]
                [k (impl/edit editor (->vals k))]))
         (into {}))))

(defn- assemble-cat [m]
  (impl/assemble-all m))

(defmethod event/handle ::edit-key [{:keys [fn on-edit key]}]
  (event/handle (assoc on-edit :fn #(update % key fn))))

(defn- view-cat [cat-keys {:keys [edit on-edit]}]
  {:fx/type impl/multi-line-view
   :children (for [k cat-keys]
               {:fx/type :h-box
                :spacing (font/char-width)
                :children [{:fx/type :label
                            :style-class "reveal-form-label"
                            :pseudo-classes #{:control}
                            :text (pr-str k)}
                           {:fx/type impl/content-view
                            :edit (edit k)
                            :on-edit {::event/type ::edit-key
                                      :key k
                                      :on-edit on-edit}}]})})

(defn cat
  "`:cat-ops` is a sequence of kvs from any key to op (either single value form
  or sequential content editor with :label)
  `:parse` needs to parse [T] to {cat-key [T]}"
  [{:keys [cat-ops parse]}]
  (let [cat-keys (map first cat-ops)
        cat-editors (map (fn [[k op]]
                           [k (op->content-editor op)])
                         cat-ops)]
    (-> (impl/make-editor
          ;; a content editor that edits a map of sequences using ops vals as editors
          :edit #(edit-cat cat-editors %)
          :assemble assemble-cat
          :view #(view-cat cat-keys %))
        (wrap-parse-content-editor
          parse
          #(into []
                 (comp
                   (map %)
                   clojure.core/cat)
                 cat-keys))
        (assoc :label (str/join "," (map pr-str cat-keys))))))

;; endregion

;; region lazy

(defn- edit-lazy [op-fn x]
  (if (impl/completely-undefined? x)
    {:undefined true}
    (let [editor (op->content-editor (op-fn))]
      {:defined (impl/edit editor x)})))

(defn- assemble-lazy [{:keys [undefined defined]}]
  (if undefined [] (impl/assemble defined)))

(defmethod event/handle ::define [{:keys [op-fn on-edit]}]
  (event/handle
    (assoc on-edit
      :fn #(-> %
               (dissoc :undefined)
               (assoc :defined (impl/edit (op->content-editor (op-fn))))))))

(defmethod event/handle ::edit-defined [{:keys [on-edit fn]}]
  (event/handle (assoc on-edit :fn #(update % :defined fn))))

(defn- view-lazy [op-fn label {:keys [edit on-edit]}]
  (let [{:keys [undefined defined]} edit]
    (if undefined
      {:fx/type :button
       :style-class "reveal-form-button"
       :text (str "+ " label)
       :on-action {::event/type ::define
                   :op-fn op-fn
                   :on-edit on-edit}}
      {:fx/type impl/content-view
       :edit defined
       :on-edit {::event/type ::edit-defined
                 :on-edit on-edit}})))

(defn lazy [{:keys [op-fn label]}]
  (assoc (impl/make-editor
           :edit #(edit-lazy op-fn %)
           :assemble assemble-lazy
           :view #(view-lazy op-fn label %))
    :label label))

;; endregion

;; region value editor

(defn- edit [kinds editor coll]
  {:kind (cond
           (impl/completely-undefined? coll) (first kinds)
           (vector? coll) :vector
           :else :list)
   :content (impl/edit editor coll)})

(defn- assemble [{:keys [kind content]}]
  (impl/fmap-result
    (impl/assemble content)
    (fn [items]
      (case kind
        :vector items
        :list (apply list items)))))

(defmethod event/handle ::set-kind [{:keys [on-edit kind]}]
  (event/handle (assoc on-edit :fn #(assoc % :kind kind))))

(defn view [kinds {:keys [edit on-edit form]}]
  (let [{:keys [kind content]} edit]
    {:fx/type impl/multi-line-view
     :children [{:fx/type vignette/view
                 :edit edit
                 :on-edit on-edit
                 :form form
                 :desc {:fx/type impl/menu-button
                        :pseudo-classes #{:object}
                        :text (case kind
                                :vector "["
                                :list "(")
                        :items (for [k kinds]
                                 {:fx/type :menu-item
                                  :on-action {::event/type ::set-kind
                                              :on-edit on-edit
                                              :kind k}
                                  :text (pr-str k)})}}
                {:fx/type impl/indent-view
                 :desc {:fx/type impl/content-view
                        :edit content
                        :on-edit {::event/type ::edit-content
                                  :on-edit on-edit}}}
                {:fx/type :label
                 :style-class "reveal-form-object"
                 :text (case kind
                         :vector "]"
                         :list ")")}]}))

(defn value-editor [{:keys [kinds op]
                     :or {kinds [:vector :list]}}]
  (let [editor (op->content-editor op)]
    (-> (impl/make-editor
          :edit #(edit kinds editor %)
          :assemble assemble
          :view #(view kinds %))
        (compatible/wrap sequential?)
        (copy-paste/wrap))))

;; endregion