(ns vlaaad.reveal.pro.form.spec-alpha
  "© 2021 Vladislav Protsenko. All rights reserved."
  (:require [clojure.spec.alpha :as s]
            [clojure.string :as str]
            [clojure.spec.gen.alpha :as sgen]
            [vlaaad.reveal.pro.form.impl :as impl]
            [vlaaad.reveal.pro.form.enum :as enum]
            [vlaaad.reveal.pro.form.tuple :as tuple]
            [vlaaad.reveal.pro.form.alt :as alt]
            [vlaaad.reveal.pro.form.text :as text]
            [vlaaad.reveal.pro.form.coll-of :as coll-of]
            [vlaaad.reveal.pro.form.map-of :as map-of]
            [vlaaad.reveal.pro.form.lazy :as lazy]
            [vlaaad.reveal.pro.form.entity :as entity]
            [vlaaad.reveal.pro.form.regex :as regex])
  (:import [java.io FileNotFoundException]))

(defn- explain [spec x]
  (when-let [explanation (s/explain-data spec x)]
    (->> explanation
         ::s/problems
         (map (fn [{:keys [pred reason in]}]
                (let [abbrev (s/abbrev pred)
                      message (or reason
                                  (when-not (= ::s/unknown abbrev)
                                    (pr-str abbrev))
                                  "invalid")]
                  (str (when (seq in)
                         (str "in " (pr-str in) ": "))
                       message))))
         (str/join "\n"))))

(defn- spec-options [spec]
  (cond-> []
    (try
      (require 'clojure.test.check.generators) true
      (catch FileNotFoundException _ false))
    (conj {:invoke #(sgen/generate (s/gen spec))
           :label "Generate"
           :shortcut [:shortcut :g]})))

(defn- distinct-by
  [f xs]
  ((fn func [acc f xs]
     (if-let [[x & xs] (seq xs)]
       (let [k (f x)]
         (if (contains? acc k)
           (recur acc f xs)
           (lazy-seq (cons x (func (conj acc k) f xs)))))
       nil))
   #{} f xs))

(defn- spec-form? [spec]
  (let [f (s/form spec)]
    (when (and (sequential? f)
               (qualified-symbol? (first f))
               (= "clojure.spec.alpha" (namespace (first f))))
      spec)))

(defn- spec-obj [spec]
  (or (s/get-spec spec) spec))

(defn- extract-keys [spec]
  (let [extract-keys-forms (fn extract-keys-forms [spec]
                             (let [f (s/form spec)]
                               (case (and (sequential? f)
                                          (first f))
                                 (clojure.spec.alpha/and
                                   clojure.spec.alpha/merge
                                   clojure.spec.alpha/nilable
                                   clojure.spec.alpha/nonconforming)
                                 (mapcat extract-keys-forms (map eval (rest f)))

                                 clojure.spec.alpha/keys
                                 (let [[_ & {:as opts}] f]
                                   [opts])

                                 [])))]
    (apply merge-with (comp distinct into) (extract-keys-forms spec))))

(defn make-form
  ([spec]
   (make-form #{} spec))
  ([visited spec]
   (make-form visited (cond-> spec (not (s/get-spec spec)) s/form) spec))
  ([visited abstract-form spec]
   (make-form visited abstract-form (s/form spec) spec))
  ([visited abstract-form concrete-form spec]
   (let [concrete (nil? (s/get-spec spec))
         abstract (not concrete)
         form (cond-> {:explain #(explain spec %)
                       :label (cond
                                (= abstract-form ::s/unknown)
                                "???"

                                abstract
                                (pr-str abstract-form)

                                (not= concrete-form ::s/unknown)
                                (pr-str (s/abbrev concrete-form))

                                :else
                                (pr-str (s/abbrev abstract-form)))
                       :options (spec-options spec)}
                (not= concrete-form ::s/unknown)
                (assoc :description (impl/pprint-str concrete-form)))
         visited' (cond-> visited abstract (conj abstract-form))
         keys->req-opt-any (fn [{:keys [req opt req-un opt-un]}]
                             (let [k->form #(make-form visited' (or (s/get-spec %) any?))
                                   unqualify #(-> % name keyword)
                                   and-or->optional-keys (fn and-or->optional-keys
                                                           ([form] (and-or->optional-keys [] form))
                                                           ([acc form]
                                                            (reduce (fn [acc el]
                                                                      (cond
                                                                        (keyword? el) (conj acc el)
                                                                        (sequential? el) (and-or->optional-keys acc el)
                                                                        :else acc))
                                                                    acc form)))]
                               {:req (distinct-by
                                       first
                                       (concat
                                         (->> req
                                              (filter keyword?)
                                              (map (juxt identity k->form)))
                                         (->> req-un
                                              (filter keyword?)
                                              (map (juxt unqualify k->form)))))
                                :opt (distinct-by
                                       first
                                       (concat
                                         (map (juxt identity k->form) opt)
                                         (map (juxt unqualify k->form) opt-un)
                                         (->> req
                                              (filter sequential?)
                                              (mapcat and-or->optional-keys)
                                              (map (juxt identity k->form)))
                                         (->> req-un
                                              (filter sequential?)
                                              (mapcat and-or->optional-keys)
                                              (map (juxt unqualify k->form)))))
                                :any k->form}))]
     (cond
       (set? concrete-form)
       (assoc form :editor (enum/value-editor {:values concrete-form}))

       (and abstract (contains? visited abstract-form))
       (assoc form :editor (lazy/value-editor
                             #(:editor (make-form #{} abstract-form concrete-form spec))))

       (sequential? concrete-form)
       (case (first concrete-form)
         clojure.spec.alpha/tuple
         (-> form
             (assoc :editor (tuple/value-editor {:forms (mapv #(make-form visited' % (eval %)) (next concrete-form))}))
             (cond-> concrete (assoc :label "tuple")))

         clojure.spec.alpha/or
         (-> form
             (assoc :editor (alt/value-editor
                              {:alts (->> concrete-form
                                          next
                                          (partition 2)
                                          (mapv (fn [[k f]]
                                                  [k (make-form visited' f (eval f))])))}))
             (cond-> concrete (assoc :label (str "(or "
                                                 (->> concrete-form
                                                      next
                                                      (partition 2)
                                                      (map first)
                                                      (str/join " "))
                                                 ")"))))

         clojure.spec.alpha/and
         (let [specs (map eval (rest concrete-form))
               item-spec (or (some spec-form? specs)
                             (first specs))]
           (recur visited abstract-form (s/form item-spec) spec))

         clojure.spec.alpha/conformer
         (let [item-spec (eval (second concrete-form))]
           (recur visited abstract-form (s/form item-spec) spec))

         clojure.spec.alpha/nilable
         (assoc form
           :editor (alt/value-editor
                     {:alts {:nil (make-form visited' nil?)
                             :val (make-form visited' (second concrete-form) (eval (second concrete-form)))}}))

         clojure.spec.alpha/nonconforming
         (recur visited abstract-form (second concrete-form) spec)

         (clojure.spec.alpha/coll-of clojure.spec.alpha/every)
         (let [[_ element-form & {:keys [kind min-count max-count count]}] concrete-form
               min-count (or min-count count)
               max-count (or max-count count)
               item-form (make-form visited' element-form (eval element-form))]
           (-> form
               (assoc :editor
                      (coll-of/value-editor
                        (cond-> {:item-form item-form
                                 :kinds (case kind
                                          clojure.core/vector? [:vector]
                                          clojure.core/list? [:list]
                                          clojure.core/sequential? [:vector :list]
                                          clojure.core/set? [:set]
                                          clojure.core/map? [:map]
                                          [:vector :set :list :map])}
                          min-count (assoc :min-count min-count)
                          max-count (assoc :max-count max-count))))
               (cond-> concrete (assoc :label (case kind
                                                clojure.core/vector? "vector"
                                                clojure.core/list? "list"
                                                clojure.core/sequential? "sequential"
                                                clojure.core/set? "set"
                                                clojure.core/map? "map"
                                                "coll")))))

         clojure.spec.alpha/every-kv
         (let [[_ k v & opts] concrete-form]
           (recur visited abstract-form (list* `s/every (list `s/tuple k v) opts) spec))

         clojure.spec.alpha/map-of
         (let [[_ k v & {:keys [min-count max-count count]}] concrete-form
               min-count (or min-count count)
               max-count (or max-count count)
               key-form (make-form visited' k (eval k))
               val-form (make-form visited' v (eval v))]
           (-> form
               (assoc :editor (map-of/value-editor
                                (cond-> {:key-form key-form
                                         :val-form val-form}
                                  min-count (assoc :min-count min-count)
                                  max-count (assoc :max-count max-count))))
               (cond-> concrete (assoc :label "map"))))

         clojure.spec.alpha/keys
         (let [[_ & {:as keys}] concrete-form]
           (-> form
               (assoc :editor (entity/value-editor (keys->req-opt-any keys)))
               (cond-> concrete (assoc :label "entity"))))

         (clojure.spec.alpha/cat
           clojure.spec.alpha/alt
           clojure.spec.alpha/&
           clojure.spec.alpha/keys*
           clojure.spec.alpha/*
           clojure.spec.alpha/+
           clojure.spec.alpha/?)
         (let [parse-repeat-fn (fn [spec]
                                 (let [item-spec (eval (second (s/form spec)))
                                       re (s/regex? (spec-obj item-spec))]
                                   (fn [xs]
                                     (->> xs
                                          (s/conform spec)
                                          (mapv #(vec
                                                   (cond-> (s/unform item-spec %)
                                                     (not re) vector)))))))]
           (-> form
               (assoc
                 :editor
                 (regex/value-editor
                   {:op
                    (letfn [;; spec op to form op
                            (make-op [visited' re-form re-spec]
                              (let [re-spec-obj (spec-obj re-spec)
                                    abstract (s/get-spec re-spec)
                                    visited'' (cond-> visited' abstract (conj re-spec))]
                                (if (and abstract (contains? visited' re-spec-obj))
                                  (regex/lazy
                                    {:op-fn #(make-op #{} re-form re-spec-obj)
                                     :label (pr-str re-spec)})
                                  (case (first re-form)
                                    clojure.spec.alpha/*
                                    (regex/* {:op (op-or-spec->regex-op-or-spec visited'' (eval (second re-form)))
                                              :parse (parse-repeat-fn re-spec-obj)})

                                    clojure.spec.alpha/+
                                    (regex/+ {:op (op-or-spec->regex-op-or-spec visited'' (eval (second re-form)))
                                              :parse (parse-repeat-fn re-spec-obj)})

                                    clojure.spec.alpha/?
                                    (let [item-form (second re-form)
                                          item-spec (eval item-form)
                                          re (s/regex? (spec-obj item-spec))]
                                      (regex/? {:op (op-or-spec->regex-op-or-spec visited'' item-spec)
                                                :parse (fn [xs]
                                                         (if (empty? xs)
                                                           []
                                                           (->> xs
                                                                (s/conform re-spec-obj)
                                                                vector
                                                                (mapv #(vec
                                                                         (cond-> (s/unform item-spec %)
                                                                           (not re) vector))))))}))

                                    clojure.spec.alpha/cat
                                    (let [pairs (partition 2 (rest re-form))
                                          k->spec (into {}
                                                        (map (fn [[k form]]
                                                               (let [s (eval form)]
                                                                 [k s])))
                                                        pairs)]
                                      (regex/cat
                                        {:cat-ops (into []
                                                        (map (fn [[k]]
                                                               [k (op-or-spec->regex-op-or-spec visited'' (k->spec k))]))
                                                        pairs)

                                         :parse (fn [xs]
                                                  (let [m (s/conform re-spec-obj xs)]
                                                    (into
                                                      {}
                                                      (map
                                                        (juxt
                                                          key
                                                          (fn [[k v]]
                                                            (let [spec (k->spec k)
                                                                  re (s/regex? (spec-obj spec))]
                                                              (vec (cond-> (s/unform spec v) (not re) vector))))))
                                                      m)))}))

                                    clojure.spec.alpha/alt
                                    (regex/alt
                                      {:alt-ops (->> re-form
                                                     rest
                                                     (partition 2)
                                                     (mapv (fn [[k f]]
                                                             [k (op-or-spec->regex-op-or-spec visited'' (eval f))])))
                                       :parse (fn [xs]
                                                [(first (s/conform re-spec-obj xs)) xs])})

                                    clojure.spec.alpha/keys*
                                    (let [[_ & {:as keys}] re-form]
                                      (regex/entity (keys->req-opt-any keys)))

                                    clojure.spec.alpha/&
                                    (if (= ::s/kvs->map (nth re-form 2 nil))
                                      (let [[_ & {:as keys}] (s/form (second (:ps re-spec-obj)))]
                                        (regex/entity (keys->req-opt-any keys)))
                                      (let [re-spec (eval (second re-form))]
                                        (make-op visited' (s/form re-spec) re-spec)))))))
                            ;; if regex op, convert to form op
                            ;; otherwise treat as standalone spec even if has regex form
                            (op-or-spec->regex-op-or-spec [visited' re-spec]
                              (let [abstract (s/get-spec re-spec)
                                    visited'' (cond-> visited' abstract (conj abstract))
                                    re-spec-obj (or abstract re-spec)]
                                (cond
                                  (and abstract (contains? visited' abstract))
                                  (regex/lazy
                                    {:op-fn #(make-op #{} (s/form re-spec-obj) re-spec-obj)
                                     :label (pr-str re-spec)})

                                  (s/regex? re-spec-obj)
                                  (make-op visited'' (s/form re-spec-obj) re-spec-obj)

                                  :else
                                  (make-form visited'' re-spec-obj))))]
                      (make-op visited' concrete-form spec))}))
               (cond-> concrete (assoc :label (name (first concrete-form))))))

         clojure.spec.alpha/merge
         (-> form
             (assoc :editor (-> spec
                                extract-keys
                                keys->req-opt-any
                                entity/value-editor))
             (cond-> concrete (assoc :label "entity")))

         clojure.spec.alpha/multi-spec
         (let [[_ mm-sym re-tag] concrete-form
               mm (eval mm-sym)
               map-like (keyword? re-tag)]
           (-> form
               (assoc
                 :editor
                 (alt/value-editor
                   {:alts (->> mm
                               methods
                               (map
                                 (juxt
                                   key
                                   (fn [[dispatch-val meth]]
                                     (let [alt-spec (meth nil)]
                                       (make-form
                                         visited'
                                         (if map-like
                                           (eval
                                             (list
                                               `s/merge
                                               (if (qualified-keyword? re-tag)
                                                 (list `s/keys :req [re-tag])
                                                 (let [kw (keyword
                                                            (str "reveal-"
                                                                 (name mm-sym)
                                                                 "-"
                                                                 (clojure.lang.RT/nextID))
                                                            (name re-tag))]
                                                   (eval (list `s/def kw #{dispatch-val}))
                                                   (list `s/keys :req-un [kw])))
                                               (s/form alt-spec)))
                                           alt-spec)))))))}))
               (cond-> concrete (assoc :label (str mm-sym)))))

         clojure.spec.alpha/fspec
         (let [[_ & {:keys [args]}] concrete-form
               sym (-> spec spec-obj meta (::s/name 'the-fn) name symbol)]
           (make-form visited abstract-form (eval (list `s/cat
                                                        :name #{(list 'quote sym)}
                                                        :args (or args
                                                                  `(s/* any?))))))

         (assoc form :editor text/value-editor))

       :else
       (assoc form :editor text/value-editor)))))
