(ns vlaaad.reveal.pro.form.entity
  "© 2021 Vladislav Protsenko. All rights reserved."
  (:require [vlaaad.reveal.pro.form.impl :as impl]
            [vlaaad.reveal.pro.form.vignette :as vignette]
            [vlaaad.reveal.pro.form.copy-paste :as copy-paste]
            [vlaaad.reveal.font :as font]
            [vlaaad.reveal.event :as event]
            [vlaaad.reveal.pro.form.compatible :as compatible])
  (:import [javafx.scene.input KeyCode KeyEvent]
           [javafx.scene.control ComboBox]))

(defn- edit-unexpected-key [x]
  {:value x})

(defn- view-unexpected-key [{:keys [edit on-edit form]}]
  {:fx/type vignette/view
   :edit edit
   :on-edit on-edit
   :form form
   :main true
   :desc {:fx/type :button
          :style-class "reveal-form-input"
          :pseudo-classes #{:object}
          :text (impl/pprint-str (:value edit))}})

(def unexpected-key-form
  {:label "???"
   :explain (constantly "unexpected key")
   :editor (copy-paste/wrap (impl/make-editor
                              :edit edit-unexpected-key
                              :assemble :value
                              :view view-unexpected-key))})

(defn- edit-content [req-keys get-form x]
  (let [m (if (impl/completely-undefined? x) {} x)
        unreq-keys (keys (apply dissoc m req-keys))
        order (into req-keys unreq-keys)]
    {:order order
     :edits (into {}
                  (map (fn [k]
                         [k (impl/edit (:editor (get-form k)) (m k impl/undefined))]))
                  order)}))

(defn- assemble-content [{:keys [order edits]}]
  (impl/fmap-result
    (impl/assemble-all (mapv edits order))
    #(mapv vector order %)))

(defmethod event/handle ::edit-key [{:keys [fn on-edit key]}]
  (event/handle (assoc on-edit :fn #(update-in % [:edits key] fn))))

(defmethod event/handle ::delete-key [{:keys [on-edit key]}]
  (event/handle (assoc on-edit :fn (fn [edit]
                                     (-> edit
                                         (update :edits dissoc key)
                                         (update :order (fn [order]
                                                          (into [] (remove #(= key %)) order))))))))

(defmethod event/handle ::add-key [{:keys [on-edit key form]}]
  (event/handle (assoc on-edit :fn (fn [edit]
                                     (-> edit
                                         (update :order conj key)
                                         (update :edits assoc key (impl/edit (:editor form))))))))

(defmethod event/handle ::open-combo-box-on-space-or-enter [{:keys [^KeyEvent fx/event]}]
  (when (#{KeyCode/SPACE KeyCode/ENTER} (.getCode event))
    (.consume event)
    (.show ^ComboBox (.getTarget event)))
  identity)

(defn- view-content [keys get-form {:keys [edit on-edit]}]
  (let [{:keys [order edits]} edit
        existing-count (count edits)
        more-keys (remove edits keys)
        more-count (count more-keys)
        at-most-count (min (- 7 existing-count) more-count)
        at-least-count (min 1 more-count)
        show-count (max at-most-count at-least-count)
        show-combo-box (< show-count more-count)
        button-count (- show-count (if show-combo-box 1 0))
        [button-keys combo-box-keys] (split-at button-count more-keys)]
    {:fx/type impl/multi-line-view
     :children
     (-> []
         (into
           (map
             (fn [k]
               {:fx/type :h-box
                :fx/key k
                :spacing (font/char-width)
                :children [{:fx/type :label
                            :style-class "reveal-form-entity-key"
                            :text (pr-str k)}
                           {:fx/type impl/form-view
                            :form (update (get-form k)
                                          :options
                                          #(-> %
                                               (or [])
                                               (conj {:invoke {::event/type ::delete-key
                                                               :key k
                                                               :on-edit on-edit}
                                                      :label "Delete Entry"
                                                      :shortcut [:alt :back-space]})))
                            :edit (edits k)
                            :on-edit {::event/type ::edit-key
                                      :key k
                                      :on-edit on-edit}}]}))
           order)
         (into
           (map (fn [k]
                  (let [form (get-form k)]
                    {:fx/type :button
                     :fx/key k
                     :style-class "reveal-form-button"
                     :on-action {::event/type ::add-key
                                 :form form
                                 :key k
                                 :on-edit on-edit}
                     :graphic {:fx/type :h-box
                               :spacing (font/char-width)
                               :children [{:fx/type :label
                                           :style-class "reveal-form-label"
                                           :pseudo-classes #{:control}
                                           :text (str "+ " (pr-str k))}
                                          {:fx/type :label
                                           :style-class "reveal-form-label"
                                           :pseudo-classes #{:hint}
                                           :text (:label form)}]}})))
           button-keys)
         (cond-> show-combo-box
           (conj
             {:fx/type impl/menu-button
              :pseudo-classes #{:control}
              :text (str "+ " (count combo-box-keys) " more keys")
              :items (for [k combo-box-keys
                           :let [form (get-form k)]]
                       {:fx/type :menu-item
                        :on-action {::event/type ::add-key
                                    :form form
                                    :key k
                                    :on-edit on-edit}
                        :graphic {:fx/type :h-box
                                  :spacing (font/char-width)
                                  :children [{:fx/type :label
                                              :style-class "reveal-form-label"
                                              :pseudo-classes #{:control}
                                              :text (pr-str k)}
                                             {:fx/type :label
                                              :style-class "reveal-form-label"
                                              :pseudo-classes #{:hint}
                                              :text (:label form)}]}})})))}))

(defn content-editor
  "Edits maps as kv vectors

  Keys:
  - :req (required) - kvs from any value to forms
  - :opt (required) - kvs from any value to forms
  - :any (optional) - fn from any key to form or nil"
  [{:keys [req opt any]}]
  (let [known (-> {}
                  (into opt)
                  (into req))
        req-keys (mapv first req)
        opt-keys (mapv first opt)
        get-form (memoize
                   (fn get-form [k]
                     (or (known k)
                         (when any (any k))
                         unexpected-key-form)))]
    (impl/make-editor
      :edit #(edit-content req-keys get-form %)
      :assemble assemble-content
      :view #(view-content (into req-keys opt-keys) get-form %))))

(defn- edit [content-editor m]
  {:content (impl/edit content-editor m)})

(defn- assemble [{:keys [content]}]
  (impl/fmap-result (impl/assemble content) #(into {} %)))

(defmethod event/handle ::edit-content [{:keys [on-edit fn]}]
  (event/handle (assoc on-edit :fn #(update % :content fn))))

(defn- view [{:keys [edit on-edit form]}]
  (let [{:keys [content]} edit]
    {:fx/type impl/multi-line-view
     :children [{:fx/type vignette/view
                 :edit edit
                 :on-edit on-edit
                 :form form
                 :main true
                 :desc {:fx/type :button
                        :style-class "reveal-form-input"
                        :pseudo-classes #{:object}
                        :text "{"}}
                {:fx/type impl/indent-view
                 :desc {:fx/type impl/content-view
                        :edit content
                        :on-edit {::event/type ::edit-content
                                  :on-edit on-edit}}}
                {:fx/type :label
                 :style-class "reveal-form-object"
                 :text "}"}]}))

(defn ^{:arglists '([{:keys [req opt any]}])} value-editor [opts]
  (let [editor (content-editor opts)]
    (-> (impl/make-editor
          :edit #(edit editor %)
          :assemble assemble
          :view view)
        (compatible/wrap map?)
        copy-paste/wrap)))

;; TODO change order of compatible/copy-paste?
;; TODO currently incompatible text forms have copy-paste options that replace the whole text...