(ns samps.match-with
  (:require [clojure.core.match :refer [match]]))

(defmacro with
  "A macro to combine matching clauses, inspired by Elixir's
  with special form[1]. The body is only evaluated if patterns
  are matched, the execution will be short-circuited otherwise.

  For example, the following statement:

      (with [{:ok v1} (do-foo)
             {:ok v2} (do-bar)]
        (do-baz v1 v2))

  Is roughly equivalent to:

      (match (do-foo)
        {:ok v1}
        (match (do-bar)
          {:ok v2}
          (do-baz v1 v2)

          on-err-2
          on-err-2)

        on-err-1
        on-err-1)

  [1]: https://hexdocs.pm/elixir/1.12/Kernel.SpecialForms.html#with/1"
  [bindings & body]
  (when-not (and (vector? bindings)
                 (not-empty bindings)
                 (even? (count bindings)))
    (throw (IllegalArgumentException. "bindings has to be a vector with even number of elements.")))
  (let [binding-pairs (partition 2 bindings)
        bindings-with-symbols (partition 2 (interleave binding-pairs (repeatedly #(gensym))))]
    (reduce (fn [body [binding-pair symbol]]
              (let [[match-pattern matchable] binding-pair]
                (if (symbol? match-pattern)
                  `(let [~match-pattern ~matchable] ~body)
                  `(let [~symbol ~matchable]
                     (match ~symbol
                       ~match-pattern
                       ~body

                       :else
                       ~symbol)))))
            `(do ~@body)
            (reverse bindings-with-symbols))))