(ns dev.polymeris.sys.http
  (:require
    [clojure.tools.logging :as log]
    [integrant.core :as ig]

    [muuntaja.core]
    [reitit.dev.pretty]
    [reitit.ring]
    [reitit.ring.coercion]
    [reitit.coercion.spec]
    [reitit.ring.middleware.exception]
    [reitit.ring.middleware.multipart]
    [reitit.ring.middleware.muuntaja]
    [reitit.ring.middleware.parameters]
    [reitit.ring.spec]
    [reitit.swagger]
    [reitit.swagger-ui]
    [ring.adapter.jetty :as jetty]
    [ring.middleware.reload]))


(def exception-middleware
  (reitit.ring.middleware.exception/create-exception-middleware
    (merge
      reitit.ring.middleware.exception/default-handlers
      {;; print stack-traces for all exceptions
       :reitit.ring.middleware.exception/wrap (fn [handler e request]
                                                (log/error e "handling" (:uri request))
                                                (handler e request))})))

(defn- handler-middleware
  [{:keys [swagger]}]
  (into (if swagger
          [reitit.swagger/swagger-feature]
          [])
        [reitit.ring.middleware.parameters/parameters-middleware
         reitit.ring.middleware.muuntaja/format-negotiate-middleware
         reitit.ring.middleware.muuntaja/format-response-middleware
         exception-middleware
         reitit.ring.middleware.muuntaja/format-request-middleware
         reitit.ring.coercion/coerce-response-middleware
         reitit.ring.coercion/coerce-request-middleware]))

(defn- handler-opts
  [{:keys [extra-opts dev] :as config}]
  (cond-> {:validate reitit.ring.spec/validate
           :exception (when dev reitit.dev.pretty/exception)
           :data {:coercion reitit.coercion.spec/coercion
                  :muuntaja muuntaja.core/instance
                  :middleware (handler-middleware config)}}
          extra-opts (merge extra-opts)))

(def ^:private swagger-routes
  ["" {:no-doc true}
   ["/swagger.json" {:get (reitit.swagger/create-swagger-handler)}]
   ["/api-docs/*" {:get (reitit.swagger-ui/create-swagger-ui-handler)}]])

(defmethod ig/prep-key ::handler
  [_ {:keys [dev swagger routes] :as config}]
  (assoc config :swagger (or swagger dev)
                :routes (if (or swagger dev) (conj routes swagger-routes) routes)))

(defmethod ig/init-key ::handler
  [_ {:keys [routes] :as config}]
  (let [handler (-> (reitit.ring/router routes (handler-opts config))
                    (reitit.ring/ring-handler (constantly {:status 404, :body "Not found"})))]
    (log/info "HTTP handler loaded")
    handler))

(defmethod ig/init-key ::server
  [_ {:keys [port join handler] :or {port 8080, join true, handler ::handler}}]
  (let [server (jetty/run-jetty handler {:port port
                                         :join? join})]
    (log/info "HTTP server started on port" port)
    server))

(defmethod ig/halt-key! ::server
  [_ server]
  (.stop server)
  (log/info "HTTP server stopped"))