(ns railjure.core
  (:require
   [cognitect.anomalies :as a]
   [railjure.protocol :refer [Failure fail?]]))

(defn failure?
  "Returns truthy iff `v` is a failure value."
  {:added "0.1"}
  [v] (if (satisfies? Failure v) (fail? v) false))

(defn ok?
  "Returns truthy iff `v` is not a failure value."
  {:inline (fn [v] `(not (failure? ~v))) :inline-arities #{1} :added "0.1"}
  [v] (not (failure? v)))

(defn fail
  "Returns a failing value with the given anomaly category and optional message
  and extra data."
  {:added "0.1"}
  ([category] (fail category nil))
  ([category message & extra-data]
   (ex-info (or message "Failure") (merge {::a/category category} extra-data))))

(defmacro catch-failure
  "Catches Exceptions and returns them, or last expr if none thrown."
  {:added "0.1"}
  [form & extra]
  `(try ~form ~@extra (catch Exception e# e#)))

(defmacro ok-let
  "Wraps values for each binding to catch exceptions and short-circuit further
  execution on all failures, including within the body. Otherwise behaves like 
  `let`. `|` can be used around fallible forms to customize handling behavior."
  {:added "0.1" :style/indent 2}
  [bindings & body]
  {:pre [(even? (count bindings))]}
  (if (zero? (count bindings))
    `(catch-failure ~@body)
    `(catch-failure
      (let ~(subvec bindings 0 2)
        (if (failure? ~(nth bindings 0))
          ~(nth bindings 0)
          (ok-let ~(subvec bindings 2) ~@body))))))

(defn- |* [direction expr body recovery]
  {:pre [(#{:first :last} direction)]}
  (let [expr-sym (gensym "expr")
        body-thread (if (= direction :first)
                      `(-> ~expr-sym ~body)
                      `(->> ~expr-sym ~body))
        new-val-sym (gensym "new-val")
        new-val-thread (if (= direction :first)
                         `(-> ~new-val-sym ~body)
                         `(->> ~new-val-sym ~body))
        retry-fn `(fn [~new-val-sym] ~new-val-thread)]
    `(ok-let [~expr-sym ~expr]
       (let [r# (catch-failure ~body-thread)]
         (if (failure? r#)
           (catch-failure (~recovery r# ~expr-sym ~retry-fn))
           r#)))))

(defmacro |
  "Wraps a threaded form to short-circuit execution if first parameter is a 
  failure, and catches exceptions to return as failures. On any failure returned,
  calls `recovery` with three args: the failure, the original threaded value, 
  and a unary fn to call the original form with a new threaded value."
  {:added "0.1"}
  ([expr body recovery]
   (|* :first expr body recovery)))

(defmacro ||
  "Wraps a threaded form to short-circuit execution if last parameter is a 
  failure, and catches exceptions to return as failures. On any failure returned, 
  calls `recovery` with three args: the failure, the original threaded value,
  and a unary fn to call the original form with a new threaded value."
  {:added "0.1"}
  ([body recovery expr]
   (|* :last expr body recovery)))

(defn !
  "Failure handler that passes failures along unchanged."
  {:added "0.1"}
  [f _expr _body] f)

(defn first-failure
  "A bare transducer (included directly in a `comp` chain, like `cat`) that 
  passes non-failures unchanged till the first failure, which it returns as 
  `reduced`. Can also be used with `reduce` directly by wrapping `f` with it."
  {:added "0.1"}
  [rf]
  (fn ([] (rf)) ([result] (rf result))
    ([result x] (if (failure? x) (rf (reduced x)) (rf result x)))))

