(ns certificaat.core
  (:require [certificaat.acme4j.account :as account]
            [certificaat.acme4j.authorization :as authorization]
            [certificaat.acme4j.certificate :as certificate]
            [certificaat.acme4j.challenge :as challenge]
            [certificaat.acme4j.registration :as registration]
            [certificaat.acme4j.session :as session]
            [certificaat.util.configuration :as c]
            [certificaat.domain :as domain]
            [certificaat.util.tentoonstelling :as t]
            [clojure.tools.logging :as log]
            [clojure.core.async :refer [<!!]]
            [clojure.set :as set]
            [clojure.spec.alpha :as s]
            [clojure.string :as str]
            [clojure.java.io :as io]
            [clojure.spec.alpha :as s])
  (:import 
    java.net.URI
    clojure.lang.ExceptionInfo
    (org.shredzone.acme4j.exception AcmeServerException AcmeUnauthorizedException AcmeRateLimitExceededException)
    org.shredzone.acme4j.Status))


(defn setup [{:keys [config-dir domain key-type key-size keypair-filename] :as options}]
  (let [account-keypair (account/keypair key-type key-size)
        domain-keypair (account/keypair key-type key-size)
        account-path (str config-dir keypair-filename) 
        domain-path (str config-dir domain "/domain.key")]
    (c/add-keypair account-path account-keypair)
    (c/add-keypair domain-path domain-keypair)
    (c/add-config options)))

(defn session [{:keys [config-dir keypair-filename acme-uri]}]
  (let [keypair (account/restore config-dir keypair-filename)]
    (session/create keypair acme-uri)))

(defn register [{:keys [config-dir keypair-filename acme-uri contact] :as options}]
  (if-let [registration-uri (c/load-uri (str config-dir "registration.uri"))]
    (let [session (session options)]
      (registration/restore session registration-uri))
    (let [keypair (account/restore config-dir keypair-filename)
          reg (registration/create keypair acme-uri contact)]
      (c/save-agreement config-dir reg)
      (registration/accept-agreement reg)
      (spit (str config-dir "registration.uri") (.getLocation reg))
      reg)))

(defn authorize [{:keys [config-dir domain san challenges]} reg]
  (let [domains (if san
                  (conj san domain)
                  [domain])]
    (for [domain domains
          :let [auth (authorization/create domain reg)]]
      [domain auth (challenge/find auth challenges)])))

(defn valid? [frozen-resource options]
  (let [session (session options)]
    (condp s/valid? (.getName (io/as-file frozen-resource))
      ::domain/registration-uri (if-let [registration-uri (c/load-uri frozen-resource)]
                            (let [registration (registration/restore session registration-uri)]
                              (domain/valid? registration)))
      ::domain/authorization-uri (if-let [authorization-uri (c/load-uri frozen-resource)]
                              (let [authorization (authorization/restore session authorization-uri)]
                                (domain/valid? authorization)))
      ::domain/certificate-uri (if-let [certificate-uri (c/load-uri frozen-resource)]
                           (let [certificate (certificate/restore session certificate-uri)]
                             (domain/valid? certificate))))))

(defn challenge [{domain :domain config-dir :config-dir :as options}]
  (let [session (session options) 
        frozen-challenges (filter (comp #(= (first %) "challenge") #(str/split % #"\.") #(.getName %)) (file-seq (io/file (str config-dir domain))))]
    (for [frozen-challenge frozen-challenges
          :let [uri (new URI (slurp frozen-challenge))
                challenge (challenge/restore session uri)]]
      (challenge/accept challenge))))

(defn pending? [frozen-resource options]
  (let [session (session options)
        authorization-uri (c/load-uri frozen-resource)
        authorization (authorization/restore session authorization-uri)]
    (= Status/PENDING (.getStatus authorization))))

(defn get-certificate [{:keys [config-dir domain organisation san] :as options} reg]
  (let [path (str config-dir domain "/")
        csr (str path "request.csr")]
    (if (.exists (io/file csr))
      (let [csrb (certificate/load-certificate-request csr)]
        (certificate/request csrb reg))
      (let [domain-keypair (account/restore path "domain.key")
            csrb (certificate/prepare domain-keypair domain organisation (when san san))]
        (certificate/persist-certificate-request csr csrb)
        (certificate/request csrb reg)))))

(defn request [{config-dir :config-dir domain :domain :as options} reg]
  (let [path (str config-dir domain "/")
        cert (get-certificate options reg)]
    (certificate/persist (str path "domain-chain.crt") cert)
    (spit (str path "certificate.uri") (.getLocation cert))
    (log/info "Well done! You will find your certificate chain in" path)))

(defn info [{config-dir :config-dir domain :domain}]
  (let [path (str config-dir domain "/")
        cert-file (str path "domain-chain.crt")
        key-file (str path "domain.key")]
    (certificate/info cert-file key-file)))

(def explain challenge/explain)







(defn exit [status msg]
  (println msg)
  (System/exit status))

(defn register [options] (register options))
(defn authorize [{config-dir :config-dir domain :domain :as options}]
  (let [reg (register options)]
    (doseq [[name auth challenges] (authorize options reg)
            i (range (count challenges))
            challenge challenges
            :let [explanation (explain challenge name)]]
      (println explanation)
      (spit (str config-dir domain "/" name "." (.getType challenge) ".challenge.txt") explanation)
      (spit (str config-dir domain "/challenge." name "." i ".uri") (.getLocation challenge))
      (spit (str config-dir domain "/authorization." name ".uri") (.getLocation auth)))))
(defn accept-challenges [options]
  (try
      (doseq [c (challenge options)
              :let [resp (<!! c)]]
        (if (= Status/VALID resp)
          (println "Well done, challenge completed.")
          (println "Sorry, challenge failed." resp)))
      (catch AcmeServerException e (exit 1 (.getMessage e)))))
(defn request [options]
  (let [reg (register options)]
    (try 
      (request options reg) ; will throw AcmeUnauthorizedException if the authorizations of some or all involved domains have expired
      (catch AcmeRateLimitExceededException e (exit 1 (.getMessage e)))
      (catch AcmeUnauthorizedException e (exit 1 (.getMessage e)))) ))

(defn run [{config-dir :config-dir domain :domain :as options}]
  (cond
    (not (valid? (str config-dir "registration.uri") options)) (register options)
    (not (valid? (str config-dir domain "/authorization." domain ".uri") options)) (authorize options)
    (or (pending? (str config-dir domain "/authorization." domain ".uri") options)
        (not (valid? (str config-dir domain "/certificate.uri") options))) (do (accept-challenges options)
                                                                               (request options))
    :else (exit 0 "Nothing left to do at this point in time.")))

(defn renew [{domain :domain config-dir :config-dir :as options}]
  (if (valid? (str config-dir domain "/authorization." domain ".uri") options)
    (request options)
    (do (authorize options)
        (accept-challenges options)
        (request options))))


