;; Copyright 2019- Zawiasa Róbert Áron.
;; SPDX-License-Identifier: Apache-2.0

(ns singular.acme
  "Automatically enable HTTPS, deploying Let's Encrypt certificates."
  (:use compojure.core)
  (:require
    [clojure.string :as str]
    [clojure.java.io :as io]
    [less.awful.ssl :as less-awful] 
    [compojure.route :as route]    
    [singular.web :as webserver]
    [singular.web.ssl :as tls]
    [certificaat.kung-fu :as cert]
    [certificaat.interface.cli :as cert-cli]
    [certificaat.util.configuration :as cert-conf]
    ))
  
(defn default-cert-configuration
  [{:keys [domain organisation contact test? config-dir san]}]
  {:config-dir (or config-dir "certificates/")
   :domain domain
   :san (or san #{})
   :keypair-filename "account.key"
   :key-type :rsa
   :key-size 2048
   :acme-uri (if test? "acme://letsencrypt.org/staging" "acme://letsencrypt.org")
   :organisation organisation
   :contact (str "mailto:admin@"domain)
   :challenges #{"http-01"}
   :hooks [:before-challenge :after-request] ; hooks to inject before challenges and after certificate request
   :plugins {:webroot {:enabled false
                       :path "/tmp"}
             :httpd {:enabled false}
             :diffie-hellman {:enabled false
                              :modulus 2048
                              :filename "dhparam.pem"
                              :group14 true}
             :email {:enabled false
                     :smtp {:host (str "smtp."domain)
                            :user "changeme"
                            :pass "changeme"
                            :port 587}
                     :sendmail false}}})

(defn get-cert
  [config]
  (cert-cli/run (merge (cert-conf/read-config config) config)))

(defn deploy
  [{:keys [config-dir]
    :as config
    :or {config-dir "certificates/"}}]
  (do
    (cert/setup (default-cert-configuration config))
    (get-cert (assoc config :config-dir config-dir))
    ))

(defn get-challenges
  [{:keys [config-dir domain]}]
  (mapv 
    slurp
    (filter 
      (comp #(= (last %) "txt") #(str/split % #"\.") #(.getName %)) 
      (file-seq (io/file (str config-dir domain))))))

(defn get-all-challenges
  [config-dir]
  (mapv 
    slurp
    (filter 
      (comp #(= (last %) "txt") #(str/split % #"\.") #(.getName %)) 
      (file-seq (io/file config-dir)))))

(defn process-challenges
  [challenges]
  (into (hash-map)
        (mapv #(let [[token secret] 
                     (->> % str/split-lines (drop 2) (drop-last 4))]
                 [(apply str (drop 11 token))
                  (apply str (drop 9 secret))]
                 ) 
              challenges)))

(defn champion
  "Complete all challenges with one sword at the end
  Integrate into your route if your webserver runs on port 80"
  [config-dir]
  (fn [request]
    (let [token (-> request :path-info (str/split #"\/") last)]
      {:status 200
       :headers {"Content-Type" "text/plain"}
       :body (get (process-challenges (get-all-challenges config-dir)) token)})))

(defn complete-challenge
  "Starts a webserver for a single champion" 
  [{:as config
    :keys [config-dir domain] 
    :or {config-dir "certificates/"}}]
  (webserver/run
    (champion config-dir) 
    {:host "0.0.0.0"
     :port 80}))

(defn guide
  "Complete the full process in a single console journey!"
  [config]
  (println "Hello there! Let's encrypt the"(:domain config)"domain address now!")
  (deploy config)
  (println "We created your certificate store ('certificates/' in default)")
  (println "ALERT! Don't forget to add that directory to your .gitignore file, if its inside a git repository!")
  (println "(Type something and hit enter)")
  (read)
  (println "Now run it:")
  (deploy config)
  (println "(Type something and hit enter)")
  (read)
  (println "Start challenge:")
  (complete-challenge config)
  (println "(Type something and hit enter)")
  (read)
  (println "Complete challenge:")
  (deploy config)
  (println "Hooray, Singular did it!") 
  )


