(ns glossa.metazoa.api
  (:require
   [glossa.metazoa.optional-deps :as deps]))

(set! *warn-on-reflection* true)

(defn- dispatch
  [imeta k]
  (let [value (:glossa.metazoa/metadata-provider (meta (get (meta imeta) k)) k)]
    value))

(defmulti render-metadata #'dispatch)

(defmulti view-metadata #'dispatch)

(defmulti check-metadata #'dispatch)

(defmulti index-for-search #'dispatch)

(defmulti query-schema #'dispatch)

(defmulti tx-data #'dispatch)

;; Default defmethods

(defmethod render-metadata :default
  [imeta k]
  (get (meta imeta) k))

(defmethod check-metadata :default
  [_imeta _k]
  ;; NOTE: This is in support of calling check-metadata on all metadata entries of a given IMeta.
  nil)

(defmethod index-for-search :default
  [_ _]
  ;; See glossa.metazoa.search.lucene
  nil)

(defprotocol Indexable
  (db-id [this] "Produce an identifier (often a symbol) for `this` that is globally unique.")
  (doc-id [this] "Produce a string identifier for `this` that is globally unique, targeted at fulltext search indexing."))

(extend-protocol Indexable
  clojure.lang.Namespace
  (db-id [ns]
    (ns-name ns))
  (doc-id [ns] (str (db-id ns)))

  clojure.lang.Var
  (db-id [vr]
    (let [{var-ns :ns var-name :name} (meta vr)]
      (symbol (name (ns-name var-ns)) (name var-name))))
  (doc-id [vr] (str (db-id vr)))

  Object
  (db-id [o]
    (let [mm (meta o)]
      (if-let [id (or (get mm :glossa.metazoa/imeta)
                      (get mm :glossa.metazoa/id))]
        id
        (gensym (str (.getName (class o)) "_")))))
  (doc-id [o] (str (db-id o))))

(def schema-check-data
  '[:and
    [:map
     [:actual {:optional true} any?]
     [:actual-err {:optional true} string?]
     [:actual-out {:optional true} string?]
     [:expected {:optional true} any?]
     [:expected-err {:optional true} any?]
     [:expected-out {:optional true} any?]
     [:satisfies {:optional true} any?]
     [:throwable {:optional true}
      [:fn {:error/fn (fn [{:keys [_value]} _] "should be a java.lang.Throwable")}
       #(instance? Throwable %)]]]
    [:fn {:error/fn (fn [{:keys [_value]} _] "check must have either :actual or :throwable")}
     #(or (contains? % :actual) (contains? % :throwable))]])

(def schema-check (eval schema-check-data))

(def check-explainer
  (when deps/malli-enabled?
    ((requiring-resolve 'malli.core/explainer) schema-check)))

(def check-validator
  (when deps/malli-enabled?
    ((requiring-resolve 'malli.core/validator) schema-check)))

(defn metadata-providers
  []
  (let [render-providers (keys (dissoc (methods render-metadata) :default))
        view-providers (keys (dissoc (methods view-metadata) :default))
        check-providers (keys (dissoc (methods check-metadata) :default))]
    {:render render-providers
     :view view-providers
     :check check-providers
     :all (distinct (concat render-providers view-providers check-providers))}))

(defn find-imetas
  ([] (find-imetas (fn [ns] (conj ((comp vals ns-publics) ns) ns))))
  ([fn-find-imetas-for-ns]
   (persistent!
    (reduce
     (fn [acc ns]
       (let [imetas (conj (fn-find-imetas-for-ns ns))]
         (reduce #(conj! %1 %2) acc imetas)))
     (transient [])
     (all-ns)))))
