(ns glossa.metazoa.query.datascript
  (:require
   [datascript.core :as d]
   [glossa.metazoa.api :as api]
   [glossa.metazoa.util :refer [logln] :as util])
  (:import
   (java.util.concurrent.atomic AtomicLong)))

(set! *warn-on-reflection* true)

(defonce temp-id
  (AtomicLong. 0))

(defn next-temp-id []
  (.decrementAndGet ^AtomicLong temp-id))

(def ^:private -conn (atom nil))

(defn metadata->tx-data
  [imetas]
  (into []
        (comp (map (juxt identity meta))
              (remove (comp nil? second))
              (map (fn [[imeta mm]]
                     (let [id (api/db-id imeta)
                           entity {:db/id (next-temp-id)
                                   :glossa.metazoa/imeta imeta
                                   :glossa.metazoa/imeta-symbol id
                                   :glossa.metazoa/imeta-type (type imeta)}
                           entity (if (var? imeta)
                                    (let [var-value (deref imeta)]
                                      (merge entity
                                             (when-not (nil? var-value)
                                               {:glossa.metazoa/imeta-value var-value
                                                :glossa.metazoa/imeta-value-type (type var-value)})))
                                    entity)]
                       (reduce-kv
                        (fn [entity mk mv]
                          (if (and (keyword? mk)
                                   (not (nil? mv)))
                            (assoc entity mk mv)
                            entity))
                        entity
                        mm)))))
        imetas))

(defn- create-connection
  [conn-atom]
  (let [schema-methods (methods api/query-schema)
        schema (reduce-kv
                (fn [schema _ schema-fn]
                  (let [schema-additions (schema-fn #_imeta nil #_k nil)]
                    (merge schema schema-additions)))
                {}
                schema-methods)]
    (reset! conn-atom (d/create-conn schema))))

(defn populate-database
  [imetas]
  (create-connection -conn)
  (d/transact! @-conn (metadata->tx-data imetas)))

(defn reset [imetas]
  (logln "Re-transacting metadata for datalog querying...")
  (populate-database imetas)
  :ok)

(defn db []
  (if-let [conn @-conn]
    @conn
    (do
      (logln "Transacting metadata for datalog querying...")
      (populate-database (api/find-imetas))
      @@-conn)))

(defn q
  [query & [maybe-db-1 maybe-db-2 :as args]]
  (let [args (if (and (d/db? maybe-db-1)
                      (d/db? maybe-db-2)
                      (= maybe-db-1 maybe-db-2))
               ;; glossa.metazoa.query/q adds a db unconditionally, to avoid
               ;; having further conditional resolve calls there. This removes
               ;; it if the user also provided one.
               (next args)
               args)]
    (apply d/q query args)))

(comment

  (reset (api/find-imetas))

  (count
   (q '[:find ?ns ?name ?added
        :in $ ?ns [?added ...]
        :where
        [?e :ns ?ns]
        [?e :name ?name]
        [?e :added ?added]]
      (db)
      (the-ns 'clojure.core)
      ["1.5" "1.6" "1.7"])))
