(ns kit.generator.git
  (:require
   [babashka.process :refer [shell]]
   [clojure.java.io :as io]
   [clojure.string :as str])
  (:import
   [java.io File FileNotFoundException]))

(def ^:dynamic *dir*)

(defn repo-root [name git-url]
  (or name
      (-> git-url
          (str/split #"/")
          (last)
          (str/split #"\.")
          (first))))

(defn repo-path [root name git-url]
  (str root File/separator (repo-root name git-url)))

; TODO: Support kit.git-config.edn?
#_(defn git-config []
    (if (.exists (clojure.java.io/file "kit.git-config.edn"))
      (read-string (slurp "kit.git-config.edn"))
      {:name "~/.ssh/id_rsa"}))

(defn join
  "Joins `head` and one or more `parts` using path separators specific to the particular
   operating system. Ignores parts that are `nil`."
  [head & parts]
  (->> parts
       (reduce (fn [path p]
                 (if p
                   (File. path p)
                   (File. path)))
               head)
       .getPath))

(defn mkdir-p
  "Creates directory `path` and any parent directories that are required.
   Returns true if it has been successful."
  [path]
  (io/make-parents (join path ".ignored")))

(defn which-git []
  (let [{:keys [out err exit]} (shell {:out :string :err :string} "which git")]
    (if (zero? exit)
      (str/trim-newline out)
      (throw (ex-info "Git not found" {:error err})))))

(defn git [& args]
  (let [command (str (which-git) " " (str/join " " args))]
    (println "$" command (when *dir* (str " # in " *dir*)))
    (let [{:keys [out err exit]} (shell {:dir *dir* :out :string :err :string} command)]
      (if (zero? exit)
        (str/trim-newline out)
        (throw (ex-info (str "Git command failed: " err) {:error err :exit exit}))))))

(defn sync-repository! [root {:keys [name url tag]} & [callback]]
  (try
    (let [path (repo-path root name url)]
      (binding [*dir* path]
        (try
          (mkdir-p path)
          (git "pull")
          (catch Exception _
            (git "clone --branch " (or tag "master") " --single-branch " url " " path))))
      (when callback (callback path)))
    (catch Exception e
      (println (.getMessage e)))
    (catch Exception e
      (println "failed to clone module:" url "\ncause:" (.getMessage e))
      (.printStackTrace e))))

(comment
  (sync-repository! "/tmp/modules" {:url "https://github.com/bilus/hirmondo" :tag "main"}))
