(ns bilus.pocketbase.sync.core
  "Core PocketBase client for Clojure using the Web API directly.

   Usage:
   ```clojure
   (require '[bilus.pocketbase.core :as pb])

   ;; Create a client
   (def client (pb/create-client \"http://localhost:8090\"))
   ```"
  (:require
   [bilus.pocketbase.schema :as schema]
   [cheshire.core :as json]
   [clj-http.client :as http]
   [into-curl.core :refer [->curl]]
   [clojure.string :as str]))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Implementation
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defrecord Client [base-url auth-store])

(defrecord Collection [^Client client collection-name])

(def ^:dynamic *debug* false)

(defn debug [& args]
  (when *debug*
    (apply println args)))

(defn full-url*
  "A full URL built by concatenating base-url and path."
  [base-url path]
  (str base-url path))

(defn store-auth!*
  "Stores authentication token and record in the auth-store of the given
   client."
  [^Client client token record]
  (reset! (:auth-store client) {:token token
                                :record record}))

(defn swapp*
  "Atomically swaps the value of atom using function f and args,
   returning old value."
  [atom f & args]
  (loop []
    (let [old @atom
          new (apply f old args)]
      (if (compare-and-set! atom old new)
        old
        (recur)))))

(defn set-token*
  "Sets the current auth token in the client's auth-store using the provided
   function, returning the old token"
  [^Client {:keys [auth-store]} new-token]
  (swapp* auth-store assoc :token new-token))

(defn request*
  "Makes an HTTP request to the PocketBase API using the client at the given path,
   using a supported HTTP method (:get, :post, :patch, :delete, etc.). Returns
   normalized response body

   Supported options:
    :body - Request body map (will be JSON encoded)
    :query-params - Query parameters map
    :headers - Additional headers map"
  ([^Client client method path]
   (request* client method path nil))
  ([^Client client method path {:keys [body query-params headers]}]
   (let [url (full-url* (:base-url client) path)
         token (get @(:auth-store client) :token)
         request-headers (cond-> {"Content-Type" "application/json"}
                           token (assoc "Authorization" token)
                           headers (merge headers))
         request {:method (str/upper-case (name method))
                  :url url
                  :headers request-headers
                  :query-params query-params
                  :as :json
                  :throw-exceptions false
                  :body (json/generate-string body)}
         {status :status response-body :body} (http/request request)]
     (debug (->curl request) "# =>" status)
     (if (<= 200 status 299)
       (schema/normalize-response response-body)
       (throw (ex-info "PocketBase API request failed"
                       {:status status
                        :body (schema/normalize-response (json/parse-string response-body))
                        :url url
                        :method method}))))))

(defn auth-store*
  "The auth store atom of the client."
  [^Client client]
  (:auth-store client))

(defn build-url*
  "A full URL built by safely concatenating the provided path to the client's base URL."
  [^Client client path]
  (full-url* (:base-url client) path))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Public
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro with-debug
  "Evaluates body with debug logging enabled."
  [& body]
  `(binding [*debug* true]
     ~@body))

(defn create-client
  "A new PocketBase client instance using a PocketBase server at the provided
   URL (e.g. \"http://localhost:8090\")."
  [base-url]
  (->Client base-url (atom {})))

(defn collection
  "Reference to a Collection instance with the given collection name."
  [^Client client collection-name]
  (->Collection client collection-name))

(defn auth-valid?
  "true if the current auth state is valid (has a token)."
  [^Client client]
  (some? (get @(auth-store* client) :token)))

(defn auth-token
  "The current auth token, or nil if not authenticated."
  [^Client client]
  (get @(auth-store* client) :token))

(defn auth-record
  "The current authenticated record data."
  [^Client client]
  (get @(auth-store* client) :record))

(defn clear-auth!
  "Clears the current auth state (logs out)."
  [^Client client]
  (reset! (auth-store* client) {}))

(defn auth-with-password
  "Authenticates a user with email/username and password, using the provided
   auth collection. If successful, returns a map with :token and :record
   fields.

   Supported options:
    :expand - Relations to expand (e.g. \"author,comments\")
    :fields - Fields to return (e.g. \"id,title,author\") "
  ([^Collection collection identity password]
   (auth-with-password collection identity password nil))
  ([^Collection {:keys [client collection-name]} identity password opts]
   (let [path (str "/api/collections/" collection-name "/auth-with-password")
         body {:identity identity
               :password password}
         query-params (select-keys opts [:expand :fields])
         {:keys [token record] :as response} (request* client :post path
                                                       {:body body
                                                        :query-params query-params})]
     (store-auth!* client token record)
     response)))

(defn auth-with-oauth2-code
  "Authenticates with an OAuth2 provider (e.g. \"google\", \"github\") and the PKCE code verifier, using
   the authorization code from OAuth2 redirect, based on the provided auth collection. If successful,
   returns a map with :token, :record, and :meta fields.

   Supported options:
    :expand - Relations to expand (e.g. \"author,comments\")
    :fields - Fields to return (e.g. \"id,title,author\")
    :create-data - Data for creating new user on sign-up "
  ([^Collection collection provider code code-verifier redirect-url]
   (auth-with-oauth2-code collection provider code code-verifier redirect-url {}))
  ([^Collection {:keys [client collection-name]} provider code code-verifier redirect-url {:keys [create-data] :as opts}]
   (let [path (str "/api/collections/" collection-name "/auth-with-oauth2")
         body {:provider provider
               :code code
               :codeVerifier code-verifier
               :redirectUrl redirect-url
               :createData create-data}
         query-params (select-keys opts [:expand :fields])
         {:keys [token record] :as response} (request* client :post path
                                                       {:body body
                                                        :query-params query-params})]
     (store-auth!* client token record)
     response)))

(defn request-otp
  "Sends a one-time password to the specified email, to authenticate
   using the provided auth collection. Returns map with :otp-id. "
  [^Collection {:keys [client collection-name]} email]
  (let [path (str "/api/collections/" collection-name "/request-otp")
        body {:email email}]
    (request* client :post path {:body body})))

(defn auth-with-otp
  "Authenticates with a one-time password received by email and the otp id, generated using `request-otp`,
   using the provided auth collection. If successful, returns a map with :token and :record keys

   Supported options:
    :expand - Relations to expand (e.g. \"author,comments\")
    :fields - Fields to return (e.g. \"id,title,author\")"
  ([^Collection collection otp-id password]
   (auth-with-otp collection otp-id password nil))
  ([^Collection {:keys [client collection-name]} otp-id password opts]
   (let [path (str "/api/collections/" collection-name "/auth-with-otp")
         body {:otpId otp-id
               :password password}
         query-params (select-keys opts [:expand :fields])
         {:keys [token record] :as response} (request* client :post path
                                                       {:body body
                                                        :query-params query-params})]
     (store-auth!* client token record)
     response)))

(defn auth-refresh
  "Refreshes the current authentication token using the provided auth collection.
   If successful, returns a map with :token and :record.

   Supported options:
    :expand - Relations to expand (e.g. \"author,comments\")
    :fields - Fields to return (e.g. \"id,title,author\") "
  ([^Collection collection]
   (auth-refresh collection nil))
  ([^Collection {:keys [client collection-name]} opts]
   (let [path (str "/api/collections/" collection-name "/auth-refresh")
         query-params (select-keys opts [:expand :fields])
         {:keys [token record] :as response} (request* client :post path
                                                       {:query-params query-params})]
     (store-auth!* client token record)
     response)))

;; TODO(bilus): Test and port to Clojurescript.
(defn verify-token
  "Verifies the provided token using the auth collection by refreshing it (the only method available in PocketBase).
   If successful, returns a map with :token and :record.

   Supported options:
    :expand - Relations to expand (e.g. \"author,comments\")
    :fields - Fields to return (e.g. \"id,title,author\") "
  ([^Collection collection token]
   (verify-token collection token nil))
  ([^Collection {:keys [client] :as collection} token opts]
   (let [old-token (set-token* client token)]
     (try
       (auth-refresh collection opts)
       (finally
         (set-token* client old-token))))))

;; TODO(Bilus): Test and port to Clojurescript.
(defn auth-with-token
  "Authenticate using the provided token using the auth collection by refreshing it (the only method available in PocketBase).
   If successful, returns a map with :token and :record. Updates the client state with the provided token, if valid.

   Supported options:
    :expand - Relations to expand (e.g. \"author,comments\")
    :fields - Fields to return (e.g. \"id,title,author\") "
  ([^Collection collection token]
   (auth-with-token collection token nil))
  ([^Collection {:keys [client] :as collection} token opts]
   (verify-token collection token opts)
   (set-token* client token)))

(defn list-auth-methods
  "Lists available authentication methods for an auth collection."
  [^Collection {:keys [client collection-name]}]
  (let [path (str "/api/collections/" collection-name "/auth-methods")]
    (->> (request* client :get path)
         (filter (fn [[_ v]] (map? v))) ;; Compatible with Clojurescript version.
         (into {}))))

(defn request-verification
  "Sends a verification email to the specified email address in the auth collection."
  [^Collection {:keys [client collection-name]} email]
  (let [path (str "/api/collections/" collection-name "/request-verification")
        body {:email email}]
    (request* client :post path {:body body})))

(defn confirm-verification
  "Confirms email in the provided auth collection with the token from the verification email."
  [^Collection {:keys [client collection-name]} token]
  (let [path (str "/api/collections/" collection-name "/confirm-verification")
        body {:token token}]
    (request* client :post path {:body body})))

(defn request-password-reset
  "Sends a password reset email to the specified address in the auth collection."
  [^Collection {:keys [client collection-name]} email]
  (let [path (str "/api/collections/" collection-name "/request-password-reset")
        body {:email email}]
    (request* client :post path {:body body})))

(defn confirm-password-reset
  "Confirms password reset for a record in the provided auth collection using the token from the reset email."
  [^Collection {:keys [client collection-name]} token password password-confirm]
  (let [path (str "/api/collections/" collection-name "/confirm-password-reset")
        body {:token token
              :password password
              :passwordConfirm password-confirm}]
    (request* client :post path {:body body})))

(defn request-email-change
  "Requests an email change for the user authenticated using the provided auth collection."
  [^Collection {:keys [client collection-name]} new-email]
  (let [path (str "/api/collections/" collection-name "/request-email-change")
        body {:newEmail new-email}]
    (request* client :post path {:body body})))

(defn confirm-email-change
  "Confirms email change with the token from the confirmation email, using the current
   account password."
  [^Collection {:keys [client collection-name]} token password]
  (let [path (str "/api/collections/" collection-name "/confirm-email-change")
        body {:token token
              :password password}]
    (request* client :post path {:body body})))

(defn impersonate
  "Impersonates a user, identified by id of their record in the provided auth collection,
   returning a map with token and record for the impersonated user. Must be authenticated
   as a superuser.

   Supported options:
    :expand - Relations to expand (e.g. \"author,comments\")
    :fields - Fields to return (e.g. \"id,title,author\")
    :duration: Token duration in seconds "
  ([^Collection collection record-id]
   (impersonate collection record-id {}))
  ([^Collection {:keys [client collection-name]} record-id {:keys [duration] :as opts}]
   (let [path (str "/api/collections/" collection-name "/impersonate/" record-id)
         body (when duration {:duration duration})
         query-params (select-keys opts [:expand :fields])
         response (request* client :post path
                            {:body body
                             :query-params query-params})]
     response)))

(defn get-file-url
  "Generates a URL for downloading a file from a record (must have :collectionId or :collectionName and :id),
   given a name of the file field or filename.

   Supported options:
    :thumb key for image thumbnails (e.g., {:thumb \"100x100\"}) "
  ([^Client client record filename]
   (get-file-url client record filename nil))
  ([^Client client record filename opts]
   (let [collection-id-or-name (or (:collectionName record) (:collectionId record))
         record-id (:id record)
         base-path (str "/api/files/" collection-id-or-name "/" record-id "/" filename)
         thumb (:thumb opts)
         path (if thumb
                (str base-path "?thumb=" thumb)
                base-path)]
     (build-url* client path))))

(comment
  (let [^Client client (create-client "http://localhost:8090")
        users (collection client "users")
        user (-> (auth-with-password users "user1@bilus.dev" "test1234"))]
    (verify-token users "eue")
    (tap> user))

;;
  )
