(ns bilus.pocketbase.records
  "PocketBase Records API wrapper for CRUD operations.

   All functions return JavaScript Promises. Use `.then` or async/await patterns.

   Usage:
   ```clojure
   (require '[bilus.pocketbase.core :as pb]
            '[bilus.pocketbase.records :as records])

   (def client (pb/create-client \"http://localhost:8090\"))

   ;; List records with pagination
   (-> (pb/collection client \"posts\")
       (records/paginated {:filter \"status = true\" :page 2 :per-page 10})
       (.then #(println \"Records:\" %)))

   ;; Get all records
   (-> (pb/collection client \"posts\")
       (records/all {:sort \"-created\"})
       (.then #(println \"All posts:\" %)))

   ;; Create a record
   (-> (records/create! client \"posts\" {:title \"Hello World\"})
       (.then #(println \"Created:\" %)))

   ;; Update a record
   (-> (records/update! client \"posts\" \"RECORD_ID\" {:title \"Updated Title\"})
       (.then #(println \"Updated:\" %)))

   ;; Delete a record
   (-> (records/delete! client \"posts\" \"RECORD_ID\")
       (.then #(println \"Deleted!\")))
   ```"
  (:require ["pocketbase" :as PocketBase]
            [bilus.pocketbase.promise :as promise]
            [bilus.pocketbase.core :refer [collection]])
  (:refer-clojure :exclude [update]))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Public
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn paginated
  "Paginated collection records.

   Supported options:
    :page - Page number (default 1)
    :per-page - Records per page (default 30)
    :filter - Filter expression (e.g., \"status = true && created > '2022-01-01'\")
    :sort - Sort expression (e.g., \"-created,id\" for DESC created, ASC id)
    :expand - Relations to expand (e.g., \"author,comments\")
    :fields - Fields to return (e.g., \"id,title,author\")
    :skip-total - Skip total count query for performance (boolean)"
  ([^js collection]
   (paginated collection {}))
  ([^js collection {:keys [page per-page] :or {page 1 per-page 30} :as opts}]
   (-> collection
       (.getList page per-page (clj->js opts))
       (promise/->clj))))

(defn all
  "Returns all records in the collection.

   Note: This fetches all records in batches. The implementation uses pagination
   with a large perPage value and skipTotal for performance.

   Supported options:
    :batch - Records per batch request (default 500)
    :filter - Filter expression
    :sort - Sort expression
    :expand - Relations to expand
    :fields - Fields to return"
  ([^js collection]
   (all collection nil))
  ([^js collection opts]
   (-> collection
       (.getFullList (clj->js opts))
       (promise/->clj))))

(defn find-first
  "The first record matching the specified filter expression (e.g., \"email = 'test@example.com'\").
   Throws if not found.

   Supported options:
    :expand - Relations to expand
    :fields - Fields to return"
  ([^js collection filter]
   (find-first collection filter nil))
  ([^js collection filter opts]
   (-> collection
       (.getFirstListItem filter (clj->js opts))
       (promise/->clj))))

(defn one
  "A single record by its ID.

   Supported options:
    :expand - Relations to expand
    :fields - Fields to return"
  ([^js collection record-id]
   (one collection record-id nil))
  ([^js collection record-id opts]
   (-> collection
       (.getOne record-id (clj->js opts))
       (promise/->clj))))

(defn create!
  "Creates a new record in the collection based on map with field values. For auth
   collections, include :password and :passwordConfirm fields.

   Supported options:
    :expand - Relations to expand in response
    :fields - Fields to return in response"
  ([^js collection body]
   (create! collection body nil))
  ([^js collection body opts]
   (-> collection
       (.create (clj->js body) (clj->js opts))
       (promise/->clj))))

(defn update!
  "Updates an existing record by its ID with provided fields.

   For auth collections changing password:
    :oldPassword (required unless superuser)
    :password
    :passwordconfirm

   Supported options:
    :expand - Relations to expand in response
    :fields - Fields to return in response"
  ([^js collection record-id body]
   (update! collection record-id body {}))
  ([^js collection record-id body opts]
   (-> collection
       (.update record-id (clj->js body) (clj->js opts))
       (promise/->clj))))

(defn delete!
  "Deletes a single record by its ID."
  [^js collection record-id]
  (-> collection
      (.delete record-id)))

(defn upsert!
  "Creates or updates a record.
   If a record with the specified ID exists, it updates it; otherwise creates new.

   Note: This is typically used via batch operations.

   Parameters:
   - collection: The collection
   - body: Map with record data (must include :id for upsert logic)
   - opts: (optional) Map with keys:
       :expand - Relations to expand in response
       :fields - Fields to return in response

   Returns: Promise resolving to the created/updated record"
  ([^js collection body]
   (upsert! collection body nil))
  ([^js collection body opts]
   ;; Note: upsert is typically used via batch, but can be called directly
   ;; via the batch's collection service
   (-> collection
       (.upsert (clj->js body) (clj->js opts))
       (promise/->clj))))

(defn subscribe
  "Subscribes to realtime changes in a collection.

   Parameters:
   - client: PocketBase client instance
   - topic: \"*\" for all records or a specific record ID
   - callback: Function called with event {:action :record}
               action is \"create\", \"update\", or \"delete\"
   - opts: (optional) Map with keys:
       :filter - Filter expression for events
       :expand - Relations to expand
       :fields - Fields to include

   Returns: Promise resolving to unsubscribe function

   Example:
   ```clojure
   ;; Subscribe to all changes in a collection
   (records/subscribe client \"posts\" \"*\"
     (fn [e]
       (println \"Action:\" (.-action e))
       (println \"Record:\" (.-record e))))

   ;; Subscribe to specific record changes
   (records/subscribe client \"posts\" \"RECORD_ID\"
     (fn [e] (println \"Record updated:\" e)))
   ```"
  ([^js collection topic callback]
   (subscribe collection topic callback nil))
  ([^js collection topic callback opts]
   (-> collection
       (.subscribe topic callback (clj->js opts)))))

(defn unsubscribe
  "Unsubscribes from realtime changes.

   Parameters:
   - collection: The collection
   - topic: (optional) \"*\" or specific record ID, or omit to unsubscribe all

   Returns: Promise"
  ([^js collection]
   (-> collection
       (.unsubscribe)))
  ([^js collection topic]
   (-> collection
       (.unsubscribe topic))))

(defn build-filter
  "Builds a filter string with placeholder parameters safely escaped.

   Parameters:
   - client: PocketBase client instance
   - expr: Filter expression with {:placeholder} syntax
   - params: Map of placeholder values

   Returns: Escaped filter string

   Example:
   ```clojure
   (records/build-filter client
     \"title ~ {:title} && status = {:status}\"
     {:title \"test\" :status true})
   ;; => \"title ~ 'test' && status = true\"
   ```"
  [^js client expr params]
  (.filter client expr (clj->js params)))

(defn create-with-form!
  "Creates a record with file uploads.

   Parameters:
   - collection: The collection
   - form-data: JavaScript FormData object with fields and files
   - opts: (optional) Map with :expand and :fields keys

   Returns: Promise resolving to the created record

   Example (in browser):
   ```clojure
   (let [form-data (js/FormData.)]
     (.append form-data \"title\" \"My Document\")
     (.append form-data \"document\" file-object)
     (records/create-with-form! collection form-data))
   ```"
  ([^js collection form-data]
   (create-with-form! collection form-data nil))
  ([^js collection form-data opts]
   (-> collection
       (.create form-data (clj->js opts))
       (promise/->clj))))

(defn update-with-form
  "Updates a record with file uploads.

   Parameters:
   - collection: The collection
   - record-id: ID of the record to update
   - form-data: JavaScript FormData object with fields and files
   - opts: (optional) Map with :expand and :fields keys

   Returns: Promise resolving to the updated record"
  ([^js collection record-id form-data]
   (update-with-form collection record-id form-data nil))
  ([^js collection record-id form-data opts]
   (-> collection
       (.update record-id form-data (clj->js opts))
       (promise/->clj))))
