(ns bilus.pocketbase.promise
  "Functions for handling Promise objects returned from PocketBase methods.")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Implementation
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; TODO: Move to cljc, clj version has a copy.
(defn transform-keys*
  "Recursively transforms all map keys using the given function."
  [f m]
  (cond
    (map? m)
    (into {} (map (fn [[k v]] [(f k) (transform-keys* f v)]) m))

    (sequential? m)
    (mapv #(transform-keys* f %) m)

    :else m))

(defn map-keys
  "Maps a function over the keys of a map."
  [f m]
  (into {} (map (fn [[k v]] [(f k) v]) m)))

(defn ->clj*
  "Converts a JS object to ClojureScript with keyword keys."
  [obj]
  (->> (js->clj obj)
       (transform-keys* keyword)))

(defn error->clj*
  "Converts a PocketBase ClientResponseError to ClojureScript map
   with :status, :message, :data keys"
  [^js error]
  {:status (.-status error)
   :message (.-message error)
   :data (->clj* (.-data error))
   :original error})

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Public
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn fmap
  "Maps a function over a Promise's success value."
  [f promise]
  (.then promise f))

(defn bimap
  "Maps functions over both success and error paths of a Promise,
   returning a Promise with transformed success/error values."
  [f-success f-error promise]
  (-> promise
      (.then f-success)
      (.catch (fn [e]
                (js/Promise.reject (f-error e))))))

(defn ->clj
  "Maps over both success and error paths of a Promise,
   returning a Promise with success and error values transformed to
  ClojureScript data."
  [promise]
  (bimap ->clj* error->clj* promise))
