(ns bilus.pocketbase.core
  "Core PocketBase client creation and authentication utilities.

   Usage:
   ```clojure
   (require '[bilus.pocketbase.core :as pb])

   ;; Create a client
   (def client (pb/create-client \"http://localhost:8090\"))

   ;; Authenticate with password
   (-> (pb/auth-with-password client \"users\" \"test@example.com\" \"password123\")
       (.then #(println \"Authenticated:\" %)))

   ;; Wrapping in async
   (go
     (let [auth-data (<p! (pb/auth-with-password client \"users\" \"\"))]
         (println \"Authenticated:\" auth-data)))

   ;; Check auth state
   (pb/auth-valid? client) ;; => true/false
   (pb/auth-token client)  ;; => \"eyJ...\"
   (pb/auth-record client) ;; => {:id \"...\" :email \"...\"}

   ;; Clear auth
   (pb/clear-auth! client)
   ```"
  (:require ["pocketbase" :as PocketBase]
            [bilus.pocketbase.promise :as promise]))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Implementation
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn auth-store*
  "Returns the auth store from the client."
  ^js [^js client]
  (.-authStore client))

(defn auth-valid?
  "true if the current auth state is valid (has a token)."
  [^js client]
  (.-isValid (auth-store* client)))

(defn auth-token
  "The current auth token, or nil if not authenticated."
  [^js client]
  (not-empty (.-token (auth-store* client))))

(defn build-url*
  "A full URL built by safely concatenating the provided path to the client's base URL."
  [^js client path]
  (.buildURL client path))

;; If true, each client gets its own private auth store instance,
;; otherwise all clients share a global auth store in local storage.
;; It is used for testing purposes.
(def ^:dynamic *private-auth-store* false)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Public
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn create-client
  "A new PocketBase client instance using a PocketBase server at the provided
   URL (e.g. \"http://localhost:8090\")."
  [base-url]
  (tap> *private-auth-store*)
  (let [auth-store-key (if *private-auth-store*
                         (str "pb_auth_store_" (random-uuid))
                         "pb_auth_store")]
    (new PocketBase/default. base-url (PocketBase/LocalAuthStore. auth-store-key))))

(defn auth-record
  "The current authenticated record data."
  [^js client]
  (-> (auth-store* client)
      (.-record)
      (js->clj :keywordize-keys true)))

(defn clear-auth!
  "Clears the current auth state (logs out)."
  [^js client]
  (.clear (auth-store* client)))

(defn auth-with-password
  "Authenticates a user with email/username and password, using the provided
   auth collection. If successful, returns a map with :token and :record
   fields.

   Supported options:
    :expand - Relations to expand (e.g. \"author,comments\")
    :fields - Fields to return (e.g. \"id,title,author\") "
  ([^js collection identity password]
   (auth-with-password collection identity password nil))
  ([^js collection identity password opts]
   (-> collection
       (.authWithPassword identity password (clj->js opts))
       (promise/->clj))))

(defn auth-with-oauth2-code
  "Authenticates with an OAuth2 provider (e.g. \"google\", \"github\") and the PKCE code verifier, using
   the authorization code from OAuth2 redirect, based on the provided auth collection. If successful,
   returns a map with :token, :record, and :meta fields.

   Supported options:
    :expand - Relations to expand (e.g. \"author,comments\")
    :fields - Fields to return (e.g. \"id,title,author\")
    :create-data - Data for creating new user on sign-up "
  ([^js collection provider code code-verifier redirect-url]
   (auth-with-oauth2-code collection provider code code-verifier redirect-url {}))
  ([^js collection provider code code-verifier redirect-url {:keys [create-data] :as opts}]
   (-> collection
       (.authWithOAuth2Code provider code code-verifier redirect-url
                            (clj->js create-data)
                            (clj->js (select-keys opts [:expand :fields])))
       (promise/->clj))))

(defn request-otp
  "Sends a one-time password to the specified email, to authenticate
   using the provided auth collection. Returns map with :otp-id. "
  [^js collection email]
  (-> collection
      (.requestOTP email)
      (promise/->clj)))

(defn auth-with-otp
  "Authenticates with a one-time password received by email and the otp id, generated using `request-otp`,
   using the provided auth collection. If successful, returns a map with :token and :record keys

   Supported options:
    :expand - Relations to expand (e.g. \"author,comments\")
    :fields - Fields to return (e.g. \"id,title,author\")"
  ([^js collection otp-id password]
   (auth-with-otp collection otp-id password nil))
  ([^js collection otp-id password opts]
   (-> collection
       (.authWithOTP otp-id password (clj->js opts))
       (promise/->clj))))

(defn auth-refresh
  "Refreshes the current authentication token using the provided auth collection.
   If successful, returns a map with :token and :record.

   Supported options:
    :expand - Relations to expand (e.g. \"author,comments\")
    :fields - Fields to return (e.g. \"id,title,author\") "
  ([^js collection]
   (auth-refresh collection nil))
  ([^js collection opts]
   (-> collection
       (.authRefresh (clj->js opts))
       (promise/->clj))))

(defn list-auth-methods
  "Lists available authentication methods for an auth collection."
  [^js collection]
  (-> collection
      (.listAuthMethods)
      (promise/->clj)))

(defn request-verification
  "Sends a verification email to the specified email address in the auth collection."
  [^js collection email]
  (-> collection
      (.requestVerification email)))

(defn confirm-verification
  "Confirms email in the provided auth collection with the token from the verification email."
  [^js collection token]
  (-> collection
      (.confirmVerification token)))

(defn request-password-reset
  "Sends a password reset email to the specified address in the auth collection."
  [^js collection email]
  (-> collection
      (.requestPasswordReset email)))

(defn confirm-password-reset
  "Confirms password reset for a record in the provided auth collection using the token from the reset email."
  [^js collection token password password-confirm]
  (-> collection
      (.confirmPasswordReset token password password-confirm)))

(defn request-email-change
  "Requests an email change for the user authenticated using the provided auth collection."
  [^js collection new-email]
  (-> collection
      (.requestEmailChange new-email)))

(defn confirm-email-change
  "Confirms email change with the token from the confirmation email, using the current
   account password."
  [^js collection token password]
  (-> collection
      (.confirmEmailChange token password)))

(defn impersonate
  "Impersonates a user, identified by id of their record in the provided auth collection,
   returning a map with token and record for the impersonated user. Must be authenticated
   as a superuser.

   Supported options:
    :expand - Relations to expand (e.g. \"author,comments\")
    :fields - Fields to return (e.g. \"id,title,author\")
    :duration: Token duration in seconds "
  ([^js collection record-id]
   (impersonate collection record-id {}))
  ([^js collection record-id {:keys [duration] :as opts}]
   (-> collection
       (.impersonate record-id duration (clj->js opts)))))

(defn create-batch
  "Creates a new batch instance for transactional operations.

   Parameters:
   - client: PocketBase client instance

   Returns: Batch instance

   Usage:
   ```clojure
   (let [batch (pb/create-batch client)]
     (-> batch (.collection \"example1\") (.create #js {:title \"test1\"}))
     (-> batch (.collection \"example2\") (.update \"RECORD_ID\" #js {:title \"test2\"}))
     (-> batch (.collection \"example3\") (.delete \"RECORD_ID\"))
     (.send batch))
   ```"
  [^js client]
  (.createBatch client))

(defn collection
  "Reference to a Collection instance with the given collection name."
  ^js [^js client collection-name]
  (.collection client collection-name))

(defn get-file-url
  "Generates a URL for downloading a file from a record (must have :collectionId or :collectionName and :id),
   given a name of the file field or filename.

   Supported options:
    :thumb key for image thumbnails (e.g., {:thumb \"100x100\"}) "
  ([client record filename]
   (get-file-url client record filename nil))
  ([^js client record filename opts]
   (.getFileUrl client record filename (clj->js opts))))
