(ns bilus.pocketbase.collections
  "PocketBase Collections API wrapper for collection management.

   Note: Most collection operations require superuser authentication.

   Usage:
   ```clojure
   (require '[bilus.pocketbase.core :as pb]
            '[bilus.pocketbase.collections :as collections])

   (def client (pb/create-client \"http://localhost:8090\"))

   ;; Authenticate as superuser first
   (-> (pb/auth-with-password client \"_superusers\" \"admin@example.com\" \"password\")
       (.then (fn [_]
                ;; List all collections
                (collections/get-full-list client))))

   ;; Create a new collection
   (-> (collections/create! client {:name \"posts\"
                                    :type \"base\"
                                    :fields [{:name \"title\" :type \"text\" :required true}
                                             {:name \"content\" :type \"editor\"}]})
       (.then #(println \"Created collection:\" %)))
   ```"
  (:require ["pocketbase" :as PocketBase]
            [bilus.pocketbase.promise :as promise]
            [bilus.pocketbase.schema :as schema])
  (:refer-clojure :exclude [import update]))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Implementation
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn collections*
  "Returns the collections service from the client."
  ^js [^js client]
  (.-collections client))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Public
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn paginated
  "A paginated collections list as a map with the following keys:
   :page :per-page :total-items :total-pages :items

   Requires superuser authentication.

   Supported options:
    :filter - Filter expression (e.g., \"type = 'auth'\")
    :sort - Sort expression (e.g., \"-created,name\")
    :fields - Fields to return
    :skip-total - Skip total count query"
  ([client]
   (paginated client 1 30 nil))
  ([client page]
   (paginated client page 30 nil))
  ([client page per-page]
   (paginated client page per-page nil))
  ([^js client page per-page opts]
   (-> (collections* client)
       (.getList page per-page (clj->js opts))
       (promise/->clj))))

(defn all
  "Returns all collections.

   Requires superuser authentication.

   Supported options:
    :batch - Collections per batch request (default 500)
    :filter - Filter expression
    :sort - Sort expression
    :fields - Fields to return"
  ([client]
   (all client nil))
  ([^js client opts]
   (-> (collections* client)
       (.getFullList (clj->js opts))
       (promise/->clj))))

(defn find-first
  "Returns the first collection matching the specified filter or throws if not found.

   Requires superuser authentication.

   Supported options:
    :fields - Fields to return"
  ([client filter]
   (find-first client filter nil))
  ([^js client filter opts]
   (-> (collections* client)
       (.getFirstListItem filter (clj->js opts))
       (promise/->clj))))

(defn one
  "Returns a single collection by its ID or name.

   Requires superuser authentication.

   Supported options:
    :fields - Fields to return"
  ([client id-or-name]
   (one client id-or-name nil))
  ([^js client id-or-name opts]
   (-> (collections* client)
       (.getOne id-or-name (clj->js opts))
       (promise/->clj))))

(defn create!
  "Creates a new collection.

   Requires superuser authentication.

   Body configuration:
    :name - (required) Unique collection name
    :type - :base (default), :view, or :auth
    :fields - Array of field definitions
    :indexes - Array of index definitions
    :system - Mark as system collection
    :list-rule, :view-rule, :create-rule, :update-rule, :delete-rule - API rules

   For view collections:
    :view-query - (required) SQL SELECT query

   For auth collections:
    :auth-rule - Auth constraint rule
    :manage-rule - Admin management rule
    :password-auth - Password auth config {:enabled :identity-fields}
    :oauth2 - OAuth2 config
    :mfa - MFA config
    :otp - OTP config

   Supported options:
    :fields - Fields for response filtering"
  ([client body]
   (create! client body nil))
  ([^js client body opts]
   (-> (collections* client)
       (.create (clj->js body) (clj->js opts))
       (promise/->clj))))

(defn update!
  "Updates an existing collection referenced by its id or name. Requires superuser authentication.

   Supported options:
   - :fields - Fields for response filtering (e.g. \"id,title,author\") TODO: Isn't this to specify fields to update?

   Example:
   ```clojure
   (collections/update! client \"posts\"
     {:list-rule \"@request.auth.id != ''\"
      :fields [(text-field :title {:required true})
               (editor-field :content)
               (text-field :category))})
   ```"
  ([client id-or-name body]
   (update! client id-or-name body nil))
  ([^js client id-or-name body opts]
   (-> (collections* client)
       (.update id-or-name (clj->js body) (clj->js opts))
       (promise/->clj))))

(defn delete!
  "Deletes a collection by its ID or name.

   Requires superuser authentication.

   Note: Will fail if the collection is referenced by other collections."
  [^js client id-or-name]
  (-> (collections* client)
      (.delete id-or-name)))

(defn truncate!
  "Deletes all records in a collection (including files and cascade relations).

   Requires superuser authentication."
  [^js client id-or-name]
  (-> (collections* client)
      (.truncate id-or-name)))

(defn import-collections!
  "Bulk imports collections configuration.

   Requires superuser authentication.

   Example:
   ```clojure
   (collections/import-collections client
     [{:name \"collection1\"
       :fields [{:name \"status\" :type \"bool\"}]}
      {:name \"collection2\"
       :fields [{:name \"title\" :type \"text\"}]}]
     false)  ; Don't delete missing collections
   ```"
  ([client collections]
   (import-collections! client collections false))
  ([^js client collections delete-missing?]
   (-> (collections* client)
       (.import (clj->js collections) delete-missing?))))

(defn scaffolds
  "Returns scaffolds (default configurations) for all collection types.

   Requires superuser authentication.

   Useful for understanding the default field structure for each collection type."
  [^js client]
  (-> (collections* client)
      (.getScaffolds)
      (promise/->clj)))

(def field-types
  "Available field types for collection schema definitions."
  {:text "text"
   :editor "editor"
   :number "number"
   :bool "bool"
   :email "email"
   :url "url"
   :date "date"
   :autodate "autodate"
   :select "select"
   :json "json"
   :file "file"
   :relation "relation"
   :password "password"})

(def collection-types
  "Available collection types."
  {:base "base"
   :view "view"
   :auth "auth"})

(defn text-field
  "Creates a text field configuration.

   Supported options:
    :required :min :max :pattern :hidden :presentable"
  ([name]
   (text-field name {}))
  ([name opts]
   (schema/merge-opts {:name name :type :text}
                      [:required :min :max :pattern :hidden :presentable] opts)))

(defn number-field
  "Creates a number field configuration.

   Supported options:
    :required :min :max :onlyInt :hidden :presentable"
  ([name]
   (number-field name {}))
  ([name opts]
   (schema/merge-opts {:name name :type :number}
                      [:required :min :max :onlyInt :hidden :presentable] opts)))

(defn bool-field
  "Creates a boolean field configuration.

   Supported options:
    :required :hidden :presentable"
  ([name]
   (bool-field name {}))
  ([name opts]
   (schema/merge-opts {:name name :type :bool}
                      [:required :hidden :presentable] opts)))

(defn email-field
  "Creates an email field configuration.

   Supported options:
    :required :exceptDomains :onlyDomains :hidden :presentable"
  ([name]
   (email-field name {}))
  ([name opts]
   (schema/merge-opts {:name name :type :email}
                      [:required :exceptDomains :onlyDomains :hidden :presentable] opts)))

(defn url-field
  "Creates a URL field configuration.

   Supported options:
    :required :exceptDomains :onlyDomains :hidden :presentable"
  ([name]
   (url-field name {}))
  ([name opts]
   (schema/merge-opts {:name name :type :url}
                      [:required :exceptDomains :onlyDomains :hidden :presentable] opts)))

(defn date-field
  "Creates a date field configuration.

   Supported options:
    :required :min :max :hidden :presentable"
  ([name]
   (date-field name {}))
  ([name opts]
   (schema/merge-opts {:name name :type :date}
                      [:required :min :max :hidden :presentable] opts)))

(defn select-field
  "Creates a select field configuration.

   Supported options:
    :required :max-select :hidden :presentable"
  ([name values]
   (select-field name values {}))
  ([name values opts]
   (schema/merge-opts {:name name :type :select :values values}
                      [:required :maxSelect :hidden :presentable] opts)))

(defn json-field
  "Creates a JSON field configuration.

   Supported options:
    :required :max-size :hidden :presentable"
  ([name]
   (json-field name {}))
  ([name opts]
   (schema/merge-opts {:name name :type :json}
                      [:required :maxSize :hidden :presentable] opts)))

(defn file-field
  "Creates a file field configuration.

   Supported options:
    :required :max-select :max-size :mime-types :thumbs :protected :hidden :presentable"
  ([name]
   (file-field name {}))
  ([name opts]
   (schema/merge-opts {:name name :type :file}
                      [:required :maxSelect :maxSize :mimeTypes :thumbs :protected :hidden :presentable] opts)))

(defn relation-field
  "Creates a relation field configuration.

   Supported options:
    :required :max-select :cascade-delete :min-select :hidden :presentable"
  ([name collection-id]
   (relation-field name collection-id {}))
  ([name collection-id opts]
   (schema/merge-opts {:name name :type :relation :collectionId collection-id}
                      [:required :maxSelect :cascadeDelete :minSelect :hidden :presentable] opts)))

(defn editor-field
  "Creates an editor (rich text) field configuration.

   Supported options:
    :required :max-size :convert-urls :hidden :presentable"
  ([name]
   (editor-field name {}))
  ([name opts]
   (schema/merge-opts {:name name :type :editor}
                      [:required :maxSize :convertUrls :hidden :presentable] opts)))
