(ns bilus.pocketbase.sync.records
  "PocketBase Records API wrapper for CRUD operations. "
  (:require
   [bilus.pocketbase.sync.core :as core]
   [bilus.pocketbase.schema :as schema])
  (:refer-clojure :exclude [update]))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Public
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn paginated
  "Paginated collection records.

   Supported options:
    :page - Page number (default 1)
    :per-page - Records per page (default 30)
    :filter - Filter expression (e.g., \"status = true && created > '2022-01-01'\")
    :sort - Sort expression (e.g., \"-created,id\" for DESC created, ASC id)
    :expand - Relations to expand (e.g., \"author,comments\")
    :fields - Fields to return (e.g., \"id,title,author\")
    :skip-total - Skip total count query for performance (boolean)

   Returns: Map with :page :per-page :total-items :total-pages :items"
  ([collection]
   (paginated collection {}))
  ([^bilus.pocketbase.sync.core.Collection {:keys [client collection-name]} {:keys [page per-page] :or {page 1 per-page 30} :as opts}]
   (let [path (str "/api/collections/" collection-name "/records")
         query-params (-> {:page page :perPage per-page}
                          (schema/merge-opts
                           [:filter :sort :expand :fields :skipTotal] opts))]
     (core/request* client :get path {:query-params query-params}))))

(defn all
  "Returns all records in the collection.

   Note: This fetches all records in batches. The implementation uses pagination
   with a large perPage value and skipTotal for performance.

   Supported options:
    :batch - Records per batch request (default 500)
    :filter - Filter expression
    :sort - Sort expression
    :expand - Relations to expand
    :fields - Fields to return"
  ([collection]
   (all collection {}))
  ([^bilus.pocketbase.sync.core.Collection {:keys [client collection-name]} opts]
   (let [batch-size (or (:batch opts) 500)]
     (loop [page 1
            all-items []]
       (let [path (str "/api/collections/" collection-name "/records")
             query-params (-> {:page page
                               :perPage batch-size
                               :skipTotal false}
                              (schema/merge-opts [:filter :sort :expand :fields] opts))
             response (core/request* client :get path {:query-params query-params})
             items (:items response)
             new-items (into all-items items)]
         (if (< (count items) batch-size)
           new-items
           (recur (inc page) new-items)))))))

(defn find-first
  "The first record matching the specified filter expression (e.g., \"email = 'test@example.com'\").
   Throws if not found.

   Supported options:
    :expand - Relations to expand
    :fields - Fields to return"
  ([collection filter]
   (find-first collection filter {}))
  ([^bilus.pocketbase.sync.core.Collection {:keys [client collection-name]} filter opts]
   (let [path (str "/api/collections/" collection-name "/records")
         query-params (-> {:page 1
                           :perPage 1
                           :skipTotal true
                           :filter filter}
                          (schema/merge-opts [:expand :fields] opts))
         response (core/request* client :get path {:query-params query-params})
         items (:items response)]
     (if (seq items)
       (first items)
       (throw (ex-info "No record found matching filter"
                       {:filter filter}))))))

(defn one
  "A single record by its ID.

   Supported options:
    :expand - Relations to expand
    :fields - Fields to return"
  ([collection record-id]
   (one collection record-id nil))
  ([^bilus.pocketbase.sync.core.Collection {:keys [client collection-name]} record-id opts]
   (let [path (str "/api/collections/" collection-name "/records/" record-id)
         query-params (-> {}
                          (schema/merge-opts [:expand :fields] opts))]
     (core/request* client :get path {:query-params query-params}))))

(defn create!
  "Creates a new record in the collection based on map with field values. For auth
   collections, include :password and :passwordConfirm fields.

   Supported options:
    :expand - Relations to expand in response
    :fields - Fields to return in response"
  ([collection body]
   (create! collection body nil))
  ([^bilus.pocketbase.sync.core.Collection {:keys [client collection-name]} body opts]
   (let [path (str "/api/collections/" collection-name "/records")
         query-params (-> {}
                          (schema/merge-opts [:expand :fields] opts))]
     (core/request* client :post path
                    {:body body
                     :query-params query-params}))))

(defn update!
  "Updates an existing record by its ID with provided fields.

   For auth collections changing password:
    :oldPassword (required unless superuser)
    :password
    :passwordconfirm

   Supported options:
    :expand - Relations to expand in response
    :fields - Fields to return in response"
  ([collection record-id body]
   (update! collection record-id body {}))
  ([^bilus.pocketbase.sync.core.Collection {:keys [client collection-name]} record-id body opts]
   (let [path (str "/api/collections/" collection-name "/records/" record-id)
         query-params (-> {}
                          (schema/merge-opts [:expand :fields] opts))]
     (core/request* client :patch path
                    {:body body
                     :query-params query-params}))))

(defn delete!
  "Deletes a single record by its ID."
  [^bilus.pocketbase.sync.core.Collection {:keys [client collection-name]} record-id]
  (let [path (str "/api/collections/" collection-name "/records/" record-id)]
    (core/request* client :delete path)
    nil))
